<?php

/***************************************************************************
	Links_DB, by Chris Marshall (chrismarshall@xemaps.com)
	
	This is a system that is meant to replace the standard Nuke Web Links system.
	It's simpler, and, in my opinion, gives a better user interface, although
	that's probably debatable. It gives a DIFFERENT user interface. Instead
	of the index-like browser interface, it gives a search-engine style interface,
	with a fairly simple and flat category system.
	
	This does not have the extensive categorization system of the standard PHP
	links system, so it's not so easy to "key" links to more than one category.
	
	There are two types of category: Single and Class.
	
	Single categories should follow a naming convention for their keys, such as:
	
	"News_Newspapers".
	
	Class Categories have a wildcard in their keys at the end, so you declare the
	the key thusly:
	
	"News_%"
	
	This will search for all categories that begin with "News_".
	
	This convention is not enforced within the code, so it's up to the super admin
	to make sure that they create categories with the correct names.
	
	Categories are listed alphabetically by their displayed name.
	
	It's VERY easy to install and administer. The administrator simply edits
	links in place, and you install this module by putting the file into your
	"modules" directory, and calling it with a super admin account the first time
	through.
	
	The Links Administrator is assigned by the standard admin control panel, and
	they need to work with the given categories, which, in turn, are only able
	to be changed by the super admin.
	
	The unique identifier for links is their URL. You can't submit more than one
	link with the same URL.
	
	The unique identifier for categories is the category key.
	
	Communication between the various forms for admin is done by form submits and
	form fields. In the standalone version.
	
	The new version allows unmoderated posting and editing of links.
	
	Like everything else I write these days, the code validates XHTML 1.1 and WAI-AAA.

	INSTALL:
	
	In order to use this file, simply create a directory called "Links_DB" in your
	"modules" directory, and place this file in there. Now that it's localized, you
	also need to place the "language" directory in the same directory as this file.
	
	As of the 2.0.21 version, you need to create and modify the "config.php" file.
	This will contain the settings for the MySQL database and the default language.
	
	Version 2.0.21 also adds a requirement that the links database be called from
	a file one step above. This allows easy integration into sites that have templates.
	
	The first time through, invoke it from a super admin account, and this script
	will create the tables.
	
	It's up to you to populate them.
	
	NOTE: This has been modified to allow standalone (not PHP-Nuke) operation. It
	should automatically detect whether or not it is installed into a Nuke directory,
	and behave accordingly.
	
	This is DBI-neutral.
	
	COPYRIGHT:
	
	As with almost all PHP-Nuke stuff, this is GPL software:
	
	This program is free software. You can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation.
	
	Copyright 2002-2005 by Chris Marshall
	
	------------------------------------------------------------------------
							VERSION HISTORY:
	------------------------------------------------------------------------
	Version 3.0.1	Bug Fix						October 4th, 2005
		Fixed a bug that caused the system to get locked into an "always submit"
		mode during administration.
		I added the "link_out" class to displayed links. This allows you to define
		a class to indicate that the link will take you away from the site.
	Version 3.0	Rework						September 20th, 2005
		This is an adjustment to ensure that the links will work as XHTML 1.1
		and validate WAI-AAA (Priority 3).
		I also now allow "mixed mode" calling (GET and POST combined).
		NOTE: The 1.1 validation means no more opening in a new window
		(good news for some people).
	Version 2.0.32	Bug Fix						July 3rd, 2004
		Fixed a bug that resulted in bogus entries sometimes being created.
		This changes the manner in which links are submitted.
	Version 2.0.31	Bug Fix						June 27th, 2004
		Fixed a bug that prevented the new subcategory system from working.
		I also found that the language strings weren't being used in one of
			the admin screens, and corrected that.
	Version 2.0.30	New Features				June 26th, 2004
		Added the code necessary to enable up to four categories per link.
	Version 2.0.29	Bug Fix						June 13th, 2004
		I added some more code to prevent invalid links from being entered.
	Version 2.0.28	Bug Fix						June 4th, 2004
		I added some code to the link submission routine to prevent invalid
			links from being added.
	Version 2.0.27	Bug Fix						May 30th, 2004
		The credit URL at the bottom of every page was broken.
	Version 2.0.26	Bug Fix						May 29th, 2004
		Fixed a few bugs in the administration system.
	Version 2.0.25	Bug Fix						May 26th, 2004
		Fixed JavaScript in the link description that puked on apostrophes.
	Version 2.0.24	Restructure					May 24th, 2004
		Changed a whole lotta stuff to make it XHTML 1.0 Transitional.
	Version 2.0.23	Bug Fix						May 23rd, 2004
		The default links per page count didn't work. Now it does.
	Version 2.0.22	Bug Fix						May 15th, 2004
		I removed the cookie mechanism. It was just too much trouble for
			not enough reward.
	Version 2.0.21	Restructure					April 28th, 2004
		I made this script much easier to include into other files.
	Version 2.0.20	Bug Fix						April 28th, 2004
		I added a timeout setting to the link tester.
	Version 2.0.19	Bug Fix						January 17th, 2004
		I fixed an open "<strong>" tag.
			I changed the "<B>...</B>" tags to "<strong>...</strong>."
	Version 2.0.18	Bug Fix						December 21st, 2003
		There was some useless text in the beginning that caused problems
			in non-nuke situations, as it referenced files in my personal
			configurations. I removed it.
		There were also some problems in the Nuke version that referenced
			files in the wrong directories.
	Version 2.0.17	Bug Fix						November 7th, 2003
		Fixed a typo in the footer.
	Version 2.0.16	Bug Fix						July 7th, 2003
		Oops. Forgot to make "$replace_URL_Header" a global.
	Version 2.0.15	Bug Fix						July 7th, 2003
		I undid the URL thingy. It interfered with too many other things, and
		I replaced it with a switch (see the "$replace_URL_Header" variable)
	Version 2.0.14	Bug Fix						July 6th, 2003
		I think that globals have been a problem. I made a fix on them.
		I fixed a problem where the check for a proper "http://" wasn't made.
		I fixed a wierd problem in which the initial category INSERT failed.
	Version 2.0.12	Bug Fix						July 6th, 2003
		I made a small fix for 6.5. I have a feeling there will need to be
			a few more.
	Version 2.0.11	Bug Fix						July 6th, 2003
		I now force the language to English if the appropriate language
			file is not present.
	Version 2.0.10	Bug Fix						April 1st, 2003
		I removed some bogus code from the main switch statement.
		I fixed a bug that sometimes resulted in strange results from
			searches in admin mode.
	Version 2.0.9	Bug Fix						December 29th, 2002
		I moved the "sql_error()" out of the PHP-Nuke test, as it's
			a non-standard function that I had added to my Nuke site.
	Version 2.0.8	Bug Fix						December 26th, 2002
		I should not display an option to sort by submitter when
			displaying outside of PHP-Nuke.
		I also made sure that the links can be edited by users only
			in the PHP-Nuke environment.
	Version 2.0.7	Bug Fix						December 25th, 2002
		I forgot to uncomment the cookie routines.
		Happy Christmas!
	Version 2.0.6	Functionality				December 22nd, 2002
		I added the ability to disable the ability to submit links.
		Fixed a few bugs with display on non-Nuke systems.
	Version 2.0.5	Bug Fix						December 22nd, 2002
		I made a few fixes for using this as a standalone system.
	Version 2.0.4	Bug Fix						December 6th, 2002
		I added a check of the "$GLOBALS" array for Nuke globals, as I think
			that new versions of PHP and Nuke 6 may not give implicit globals.
	Version 2.0.3	Bug Fix						November 13th, 2002
		I fixed a bug in which the search string was not working properly.
		I renamed a couple of functions that interfered with PHP-Nuke.
		I now properly call the header and footer functions.
		I added a bit of JavaScript to clear the search string when selecting
			a new category. Nice cosmetics.
	Version 2.0.2	Bug Fix						October 5th, 2002
		I fixed a bug in which the "More/Fewer Options" weren't working, and
			another one, where some default values were overriding the ones in the cookie.
	Version 2.0.1	Bug Fix						October 4th, 2002
		I added a function to the generic database stuff to return an SQL error.
		I fixed a problem where I forgot to "globalize" $user_prefix.
		This will not work completely as a standalone system. This is because of the
			dependence upon the PHP-Nuke user database. If anyone feels like addressing
			that, they are welcome, but it's not really worth it for me. I did the lazy thing,
			and just made it so none of the user stuff is called if not in Nuke.
	Version	2.0		Functionality				October 3rd, 2002
		I have *theoretically* added the ability to use this system outside
			of PHP-Nuke. In order to do this, I pasted in all of the SQL functions
			from PHP-Nuke 5.6.
		I added a "state" cookie. This keeps things such as the current search
			and number of links per page. It makes the system completely persistent.
		If the user doesn't have cookies enabled, it should be OK, but Nuke doesn't
			work without cookies anyway.
	Version 1.2.0	Functionality				September 11, 2002
		I now add the ability to sort out a category when you click on the
			category description. When you do searches, or select
			wildcard categories, you get category descriptions next to each link.
			An alert now asks if you want to see all of the links in just the one
			category, where it used to just tell you about that category.
		I now use the new-fangled $_GET and $_POST superglobals.
		I added the ability for individual users to add and edit their own links
			(no moderation system).
		I made the system localizable.
		I added the structure for having "subcategories" in links, but I have not
			yet added the full functionality for editing the subcategories.
		STILL TO DO:	1) Edit Link Subcategories Functionality
						2) Initial French localization (will need review by French speaker).
	Version 1.1.8	Bug Fix						September 9, 2002
		The edit fields in the add and edit pages weren't big enough,
			and the tester didn't work when you selected=\"selected\" all links.
	Version 1.1.7	Bug Fix						September 8, 2002
		I now set the initial category of a new link to the selected=\"selected\"
			one, unless the selected=\"selected\" one is a "catchall".
		I added a default sort order specifier.
	Version 1.1.6	Bug Fix						September 5, 2002
		The submit code needed to test for the URL BEFORE trying
			to submit it.
	Version 1.1.5	Bug Fix						August 24, 2002
		The categories weren't displaying if you selected=\"selected\" "Display
			Every Category".
	Version 1.1.4	Bug Fix						August 20, 2002
		I fixed an annoying bug that would display nothing after
			an administrator submitted an inactive link.
	Version 1.1.3	Bug Fix						August 14, 2002
		I fixed a syntax error in the table creation routine.
	Version 1.1.2	Bug Fix						August 9, 2002
		I fixed a problem in which the "Display Every Category" link didn't
			work properly.
	Version 1.1.1	Slight Mod					August 1, 2002
		You can now specify a default category by key.
		I added the ability to sort and search by the submitter.
	Version 1.1		Fixes and Functionality		August 1, 2002
		The reporting for the links checker was not being displayed. I
			fixed it by displaying it at the bottom of the main page.
		The editor/user method was not being applied because the globals weren't
			being carried into the functions.
		I added the ability for registered users to submit links
		I made the add new link process much simpler and more effective. It
			no longer creates an empty cell first.
		I now associate links with the users that submitted them.
		The links now have "active" status. This is used to allow users
			to submit links.
		The admin now uses the same mechanism for adding links as the users.
		I now create the default category (local).
		I now let you pick the number of pages to be displayed at the bottom
			of the links display.
	Version 1.0		Initial Release				July 31, 2002
*/

/***************************************************************************
	These are globals imported from outside namespaces and used throughout.
*/
global $php_nuke_environment, $currentlang, $dbi, $dbhost, $dbuname, $dbpass, $dbname, $dbtype, $replace_URL_Header, $users_can_submit, $moderate_links, $browserstate, $http_vars, $def_count, $my_UserID, $CPASS, $LOGIN;

require_once("Links_DB/config.php");

if ( !$currentlang )
	{
	$currentlang = "english";
	}

// We try to get the correct form variables

// We get both GET and POST parameters.

$http_vars = $_GET;

$http_vars['LOGIN'] = "";	// For security reasons, we never let these be GET parameters.
$http_vars['CPASS'] = "";

while ( list ( $key_get, $value_get ) = each ( $_POST ) )
	{
	// POST can always overwrite GET
	$http_vars[$key_get] = $value_get;
	}

if ( $php_nuke_environment )
	{
	if (!eregi("modules.php", $PHP_SELF))
		{
		die ("You can't access this file directly...");
		}
	
	$mylang = $currentlang;
	
	if ( !file_exists("modules/".basename(dirname(__FILE__))."/language/lang-$mylang.php") )
		$mylang = "english";
	
	// Get the language module
	require_once("modules/".basename(dirname(__FILE__))."/language/lang-$mylang.php");
	
	global $index, $prefix, $user, $admin;
	
	if ( !$index )
		{
		$index = $GLOBALS["index"];
		}
	
	if ( !$prefix )
		{
		$prefix = $GLOBALS["prefix"];
		}
	
	if ( !$admin )
		{
		$admin = $GLOBALS["admin"];
		}
	
	if ( !$user )
		{
		$user = $GLOBALS["user"];
		}
	
	$index			= 1;				// Indicates this is an index module
	$target_File	= "modules.php";	// Nuke always needs this file as the form target
	}
else
	{
	// Get the language module
	
	$mylang = $currentlang;
	
	// Get the language module
	require_once("./Links_DB/language/lang-$mylang.php");

	$target_File	= "./index.php";	// No-nuke uses this file as the form target

	$print_header		= "";	// Set to nonzero to print the header (only for non-Nuke.)
	$print_footer		= "";	// Set to nonzero to print the footer (only for non-Nuke.)
	}

/***************************************************************************
	These are internal globals that define the basic configuration.
*/
$c_LinksPrefix		= "_Links_DB";					// Our own prefix for the tables
$c_LinksTable		= $prefix.$c_LinksPrefix;			// The name of the links table
$c_CategoryTable	= $prefix.$c_LinksPrefix."_Categories";	// The name of the categories table
$mod_Name		= "Links_DB";					// The name of this module
$def_count			= 10;						// The number of links per page in basic view
$editormethod		= "post";						// The form method for administration
$usermethod		= "get";						// The form method for users
$page_threshold		= 11;						// The number of pages to be displayed at the bottom
												// If there are more than this number of pages, only
												// this number are shown, and parenthesized numbers
												// are shown at the ends (first page) n n n (last page).
$default_category	= "";							// If no specific category is selected=\"selected\", use this.
												// Leave it blank if you want to display all categories.
												// Make it the category _key for a default category.
$instructions_url	= "";							// The URL for an instructions page.
$default_sort		= "Timestamp";					// If no sort order is specified, use this.

/***************************************************************************
	Beginning of functions (main code is at the end of the file)
*/

/***************************************************************************
	If this is NOT the PHP-Nuke environment, then we need to add all this
	stuff to act as an SQL abstraction layer. It's implcitly included in
	PHP-Nuke.
	Notice that this stuff is right out of PHP-Nuke 5.6, and is covered by
	the copyrights and GPL license therein.
*/
if ( !$php_nuke_environment )
{
	// These are dummy functions for when there is no PHP-Nuke
	function is_user ( $user )
		{
		return 1;
		}

	function getusrinfo ( $user )
		{
		return "";
		}

	/************************************************************************/
	/* PHP-NUKE: Web Portal System                                          */
	/* ===========================                                          */
	/*                                                                      */
	/* Copyright (c) 2002 by Francisco Burzi                                */
	/* http://phpnuke.org                                                   */
	/*                                                                      */
	/* PostgreSQL fix by Rubn Campos - Oscar Silla                         */
	/*                                                                      */
	/* This program is free software. You can redistribute it and/or modify */
	/* it under the terms of the GNU General Public License as published by */
	/* the Free Software Foundation; either version 2 of the License.       */
	/************************************************************************/
	
	/* $dbtype = "MySQL"; */
	/* $dbtype = "mSQL"; */
	/* $dbtype = "PostgreSQL"; */
	/* $dbtype = "PostgreSQL_local";// When postmaster start without "-i" option. */
	/* $dbtype = "ODBC"; */
	/* $dbtype = "ODBC_Adabas"; */
	/* $dbtype = "Interbase"; */
	/* $dbtype = "Sybase"; */
	
	/*
	 * sql_connect($host, $user, $password, $db)
	 * returns the connection ID
	 */
	
	class ResultSet {
		var $result;
		var $total_rows;
		var $fetched_rows;
	
		function set_result( $res ) {
			$this->result = $res;
		}
	
		function get_result() {
			return $this->result;
		}
	
		function set_total_rows( $rows ) {
			$this->total_rows = $rows;
		}
	
		function get_total_rows() {
			return $this->total_rows;
		}
	
		function set_fetched_rows( $rows ) {
			$this->fetched_rows = $rows;
		}
	
		function get_fetched_rows() {
			return $this->fetched_rows;
		}
	
		function increment_fetched_rows() {
			$this->fetched_rows = $this->fetched_rows + 1;
		}
	}
	
	
	
	function sql_connect($host, $user, $password, $db)
	{
	global $dbtype;
	switch ($dbtype) {
	
		case "MySQL":
			$dbi=@mysql_connect($host, $user, $password);
			mysql_select_db($db);
			return $dbi;
		break;
	
		case "mSQL":
			 $dbi=msql_connect($host);
			 msql_select_db($db);
			 return $dbi;
		break;
	
		case "PostgreSQL":
			 $dbi=@pg_connect("host=$host user=$user password=$password port=5432 dbname=$db");
			 return $dbi;
		break;
	
		case "PostgreSQL_local":
			 $dbi=@pg_connect("user=$user password=$password dbname=$db");
			 return $dbi;
		break;
	
		case "ODBC":
			 $dbi=@odbc_connect($db,$user,$password);
			 return $dbi;
		break;
	
		case "ODBC_Adabas":
			 $dbi=@odbc_connect($host.":".$db,$user,$password);
		 return $dbi;
		break;
	
		case "Interbase":
			 $dbi=@ibase_connect($host.":".$db,$user,$password);
			 return $dbi;
		break;
	
		case "Sybase":
			$dbi=@sybase_connect($host, $user, $password);
			sybase_select_db($db,$dbi);
			return $dbi;
		break;
	
		default:
		break;
		}
	
	}
	
	function sql_logout($id)
	{
	global $dbtype;
	switch ($dbtype) {
	
		case "MySQL":
			$dbi=@mysql_close($id);
			return $dbi;
		break;
	
		case "mSQL":
			 $dbi=@msql_close($id);
			 return $dbi;
		break;
	
		case "PostgreSQL":
		case "PostgreSQL_local":
			 $dbi=@pg_close($id);
			 return $dbi;
		break;
	  
		case "ODBC":
		case "ODBC_Adabas":
			 $dbi=@odbc_close($id);
			 return $dbi;  
		break;
	
		case "Interbase":
			 $dbi=@ibase_close($id);
			 return $dbi;
		break;
	
		case "Sybase":
			$dbi=@sybase_close($id);
			return $dbi;
		break;
	
		default:
		break;
		}
	}
	
	
	/*
	 * sql_query($query, $id)
	 * executes an SQL statement, returns a result identifier
	 */
	
	function sql_query($query, $id)
	{
	
	global $dbtype;
	global $sql_debug;
	$sql_debug = 0;
	if($sql_debug) echo "SQL query: ".str_replace(",",", ",$query)."<br />";
	switch ($dbtype) {
	
		case "MySQL":
			$res=@mysql_query($query, $id);
			return $res;
		break;
	
		case "mSQL":
			$res=@msql_query($query, $id);
			return $res;
		break;
	
		case "PostgreSQL":
		case "PostgreSQL_local":
		$res=pg_exec($id,$query);
		$result_set = new ResultSet;
		$result_set->set_result( $res );
		$result_set->set_total_rows( sql_num_rows( $result_set ) );
		$result_set->set_fetched_rows( 0 );
			return $result_set;
		break;
	
		case "ODBC":
		case "ODBC_Adabas":
			$res=@odbc_exec($id,$query);
			return $res;
		break;
	
		case "Interbase":
			$res=@ibase_query($id,$query);
			return $res;
		break;
	
		case "Sybase":
			$res=@sybase_query($query, $id);
			return $res;
		break;
	
		default:
		break;
	
		}
	}
	
	/*
	 * sql_num_rows($res)
	 * given a result identifier, returns the number of affected rows
	 */
	
	function sql_num_rows($res)
	{
	global $dbtype;
	switch ($dbtype) {
	 
		case "MySQL":
			$rows=mysql_num_rows($res);
			return $rows;
		break;
	
		case "mSQL":  
			$rows=msql_num_rows($res);
			return $rows;
		break;
	
		case "PostgreSQL":
		case "PostgreSQL_local":
			$rows=pg_numrows( $res->get_result() );
			return $rows;
		break;
			
		case "ODBC":
		case "ODBC_Adabas":
			$rows=odbc_num_rows($res);
			return $rows; 
		break;
			
		case "Interbase":
		echo "<br />Error! PHP dosen't support ibase_numrows!<br />";
			return $rows; 
		break;
	
		case "Sybase":
			$rows=sybase_num_rows($res);
			return $rows; 
		break;
	
		default:
		break;
		}
	}
	
	/*
	 * sql_fetch_row(&$res,$row)
	 * given a result identifier, returns an array with the resulting row
	 * Needs also a row number for compatibility with PostgreSQL
	 */
	
	function sql_fetch_row(&$res, $nr)
	{
	global $dbtype;
	switch ($dbtype) {
	
		case "MySQL":
			$row = mysql_fetch_row($res);
			return $row;
		break;
	
		case "mSQL":
			$row = msql_fetch_row($res);
			return $row;
		break;
	
		case "PostgreSQL":
		case "PostgreSQL_local":
		if ( $res->get_total_rows() > $res->get_fetched_rows() ) {
			$row = pg_fetch_row($res->get_result(), $res->get_fetched_rows() );
			$res->increment_fetched_rows();
			return $row;
		} else {
			return false;
		}
		break;
	
		case "ODBC":
		case "ODBC_Adabas":
			$row = array();
			$cols = odbc_fetch_into($res, $nr, $row);
			return $row;
		break;
	
		case "Interbase":
			$row = ibase_fetch_row($res);
			return $row;
		break;
	
		case "Sybase":
			$row = sybase_fetch_row($res);
			return $row;
		break;
	
		default:
		break;
		}
	}
	
	/*
	 * sql_fetch_array($res,$row)
	 * given a result identifier, returns an associative array
	 * with the resulting row using field names as keys.
	 * Needs also a row number for compatibility with PostgreSQL.
	 */
	
	function sql_fetch_array(&$res, $nr)
	{
	global $dbtype;
	switch ($dbtype)
		{
		case "MySQL":
			$row = array();
			$row = mysql_fetch_array($res);
			return $row;
		break;
	
		case "mSQL":
			$row = array();
			$row = msql_fetch_array($res);
			return $row;
		break;
	
		case "PostgreSQL":
		case "PostgreSQL_local":
		if( $res->get_total_rows() > $res->get_fetched_rows() ) {
			$row = array();
			$row = pg_fetch_array($res->get_result(), $res->get_fetched_rows() );
			$res->increment_fetched_rows();
			return $row;
		} else {
			return false;
		}
		break;
	
	/*
	 * ODBC doesn't have a native _fetch_array(), so we have to
	 * use a trick. Beware: this might cause HUGE loads!
	 */
	
		case "ODBC":
			$row = array();
			$result = array();
			$result = odbc_fetch_row($res, $nr);
		$nf = odbc_num_fields($res); /* Field numbering starts at 1 */
			for($count=1; $count < $nf+1; $count++)
		{
				$field_name = odbc_field_name($res, $count);
				$field_value = odbc_result($res, $field_name);
				$row[$field_name] = $field_value;
			}
			return $row;
		break;
	
		case "ODBC_Adabas":
			$row = array();
			$result = array();
			$result = odbc_fetch_row($res, $nr);
	
			$nf = count($result)+2; /* Field numbering starts at 1 */
		for($count=1; $count < $nf; $count++) {
			$field_name = odbc_field_name($res, $count);
			$field_value = odbc_result($res, $field_name);
			$row[$field_name] = $field_value;
		}
			return $row;
		break;
	
		case "Interbase":
		$orow=ibase_fetch_object($res);
		$row=get_object_vars($orow);
			return $row;
		break;
	
		case "Sybase":
			$row = sybase_fetch_array($res);
			return $row;
		break;
	
		}
	}
	
	function sql_fetch_object(&$res, $nr)
	{
	global $dbtype;
	switch ($dbtype)
		{
		case "MySQL":
			$row = mysql_fetch_object($res);
		if($row) return $row;
		else return false;
		break;
	
		case "mSQL":
			$row = msql_fetch_object($res);
		if($row) return $row;
		else return false;
		break;
	
		case "PostgreSQL":
		case "PostgreSQL_local":
		if( $res->get_total_rows() > $res->get_fetched_rows() ) {
			$row = pg_fetch_object( $res->get_result(), $res->get_fetched_rows() );
			$res->increment_fetched_rows();
			if($row) return $row;
			else return false;
		} else {
			return false;
		}
		break;
	
		case "ODBC":
			$result = odbc_fetch_row($res, $nr);
		if(!$result) return false;
		$nf = odbc_num_fields($res); /* Field numbering starts at 1 */
			for($count=1; $count < $nf+1; $count++)
		{
				$field_name = odbc_field_name($res, $count);
				$field_value = odbc_result($res, $field_name);
				$row->$field_name = $field_value;
			}
			return $row;
		break;
	
		case "ODBC_Adabas":
			$result = odbc_fetch_row($res, $nr);
		if(!$result) return false;
	
			$nf = count($result)+2; /* Field numbering starts at 1 */
		for($count=1; $count < $nf; $count++) {
			$field_name = odbc_field_name($res, $count);
			$field_value = odbc_result($res, $field_name);
			$row->$field_name = $field_value;
		}
			return $row;
		break;
	
		case "Interbase":
			$orow = ibase_fetch_object($res);
		if($orow)
		{
			$arow=get_object_vars($orow);
			while(list($name,$key)=each($arow))
			{
			$name=strtolower($name);
			$row->$name=$key;
			}
				return $row;
		}else return false;
		break;
	
		case "Sybase":
			$row = sybase_fetch_object($res);
			return $row;
		break;
	
		}
	}
	
	/*** Function Free Result for function free the memory ***/
	function sql_free_result($res) {
	global $dbtype;
	switch ($dbtype) {
	
		case "MySQL":
			$row = mysql_free_result($res);
			return $row;
		break;
	
		   case "mSQL":
			$row = msql_free_result($res);
			return $row;
		break;
	
	
			case "PostgreSQL":
		case "PostgreSQL_local":
			$rows=pg_FreeResult( $res->get_result() );
			return $rows;
		break;
	
		case "ODBC":
		case "ODBC_Adabas":
			$rows=odbc_free_result($res);
			return $rows;
		break;
	
		case "Interbase":
		echo "<br />Error! PHP dosen't support ibase_numrows!<br />";
			return $rows;
		break;
	
		case "Sybase":
			$rows=sybase_free_result($res);
			return $rows;
		break;
		}
	}

	$dbi = sql_connect( $dbhost, $dbuname, $dbpass, $dbname );
//	echo "SQL Connection: \$dbhost = \"$dbhost\", \$dbuname = \"$dbuname\", \$dbpass = \"$dbpass\", \$dbname = \"$dbname\", \$dbi = \"$dbi\"<br />\n";
}

if ( !function_exists ( sql_error ) )
	{
	// Added by Chris Marshall -Just so we can get an error if it's available. Very helpful.
	function sql_error ()
		{
		global $dbtype;
		switch ($dbtype) {
		
			case "MySQL":
				return htmlentities ( mysql_error() );
			break;
		
			case "mSQL":
				return htmlentities ( msql_error() );
			break;
		
			case "ODBC":
			case "ODBC_Adabas":
				return htmlentities ( odbc_error() );
			break;
		
			default:
				return "";
			break;
			}
		}
	}

/***************************************************************************
	Displays the header for this page
*/
function Links_HeaderDisplay ( )
	{
	global $php_nuke_environment, $print_header;
	
	if ( $php_nuke_environment )
		{
		include("Links_DB/header.php");
		OpenTable();
		}
	else
		{
		if ( $print_header )
			{
			}
		}
	}
	
/***************************************************************************
	Displays the footer for this page
*/
function Links_FooterDisplay ( )
	{
	global $php_nuke_environment, $print_footer;
	
	echo "<br /><div style=\"text-align: center\"><span style=\"font-size:x-small\"><a href=\"http://www.cmarshall.net/Links_DB\">This is an implementation of the open-source Links DB System</a></span></div>";
	
	if ( $php_nuke_environment )
		{
		CloseTable();
		include("Links_DB/footer.php");
		}
	else
		{
		if ( $print_footer )
			{
			}
		}
	}

/***************************************************************************
	This function returns true if the current admin user is cleared for
	the given admin function:
	
	"EditLinks"			-> Edit the links in the database (Links Admin)
	"EditCategories"	-> Edit the link categories	(Super Admin)
*/
function can_admin ( $admin_function )
	{
	global $php_nuke_environment, $users_can_submit, $my_UserID;
	
	// The following admin test only works in the PHP-Nuke system
	if ( $php_nuke_environment )
		{
		global $prefix, $admin, $dbi;
		
		if(!is_array($admin))
			{
			$adminSpec = explode(":", base64_decode($admin));
			}
		else
			{
			$adminSpec = $admin;
			}
	
		$aid = $adminSpec[0];
		$pwd = $adminSpec[1];
	
		$result = sql_query("select radminlink, radminsuper from ".$prefix."_authors where aid='$aid'", $dbi);
		if ( $result && (sql_num_rows ( $result, $dbi ) > 0) )
			{
			list($radminlink, $radminsuper) = sql_fetch_row($result, $dbi);
			
			if ( $radminlink && ($admin_function == "EditLinks") )
				{
				return 1;
				}
			elseif ( $radminsuper )	// Super admin can do anything
				{
				return 1;
				}
			}
		}
	else
		{
		if ( $users_can_submit || ($my_UserID == 1)
		|| ($my_UserID && ValidateUser ( $my_UserID, $admin_function )) )
			{
			return 1;
			}
		else
			{
			return "";
			}
		}
	
	return "";
	}

/***************************************************************************
	This function returns true if the current user is cleared to submit links.
*/
function can_submit ( )
	{
	global $users_can_submit, $php_nuke_environment, $user;
	
	if ( can_admin ( "EditLinks" ) )
		{
		return 1;
		}
	elseif ( $users_can_submit )
		{
		if ( $php_nuke_environment )
			{
			if ( is_user ( $user ) )
				{
				return 1;
				}
			}
		else
			{
			}
		}
	
	return "";
	}

/***************************************************************************
	These are all admin functions, and don't even get included if we aren't
	an administrator
*/

if ( can_admin ( "EditLinks" ) )	// All administrators can see these
	{
	/***************************************************************************
		Deletes all the links in the found set
	*/
	function DeleteFoundLinks ( $http_vars )
		{
		global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;
	
		if ( can_admin ( "EditLinks" ) )
			{
			$whereClause = "";
			
			if ( $http_vars[SEARCH_STRING] )
				{
				$whereClause = "WHERE ((_url LIKE '%".$http_vars[SEARCH_STRING]."%')";
				$whereClause .= " OR (_title LIKE '%".$http_vars[SEARCH_STRING]."%')";
				$whereClause .= " OR (_category LIKE '%".$http_vars[SEARCH_STRING]."%')";
				
				$sql = "SELECT _key FROM $c_CategoryTable WHERE _name LIKE '%".$http_vars[SEARCH_STRING]."%'";
	
				$result = sql_query ( $sql, $dbi );
				
				if ( $result )
					{
					while ( $row = sql_fetch_array ( $result, $dbi ) )
						{
						$whereClause .= " OR (_category = '$row[_key]')";
						}
					}
				
				$whereClause .= " OR (_description LIKE '%".$http_vars[SEARCH_STRING]."%'))";
				}
			elseif ( $http_vars[FIND_CAT] )
				{
				if ( preg_match ( "/(.*)\%$/", $http_vars[FIND_CAT] ) )
					{
					$whereClause = "WHERE (_category LIKE '".$http_vars[FIND_CAT]."%')";
					}
				else
					{
					$whereClause = "WHERE (_category='$http_vars[FIND_CAT]')";
					}
				}
				
			$sql = "DELETE FROM $c_LinksTable $whereClause";
			
			$result = sql_query ( $sql, $dbi );
			
			if ( $result )
				{
				print ( "<h2>"._LINKS_DELETED."</h2><br />\n" );
				}
			else
				{
				print ( "<h2>"._ERROR._NO_LINKS_DELETED."</h2><br />\n" );
				}
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		
		$http_vars[LINK_ACTION] = "";
		DisplayLinks ( $http_vars );
		}

	/***************************************************************************
		Activates a link
	*/
	function Activate ( $http_vars )
		{
		global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;
	
		if ( can_admin ( "EditLinks" ) )
			{
			$sql = "UPDATE $c_LinksTable SET _active='1' WHERE _url='$http_vars[LINKURL]'";
			$result = sql_query ( $sql, $dbi );
			if ( $result )
				{
				print ( "<h2>"._ACTIVATED."</h2>" );
				}
			else
				{
				print ( "<h2>"._ERROR._ACTIVATED_FAILED."</h2>" );
				}
			
			DisplayLinks ( $http_vars );
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		}
	
	/***************************************************************************
		Deactivates a link
	*/
	function Deactivate ( $http_vars )
		{
		global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;
	
		if ( can_admin ( "EditLinks" ) )
			{
			$sql = "UPDATE $c_LinksTable SET _active='0' WHERE _url='$http_vars[LINKURL]'";
			$result = sql_query ( $sql, $dbi );
			if ( $result )
				{
				print ( "<h2>"._DEACTIVATED."</h2>" );
				}
			else
				{
				print ( "<h2>"._ERROR._DEACTIVATED_FAILED."</h2>" );
				}
			
			DisplayLinks ( $http_vars );
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		}
	
	/***************************************************************************
		Runs a basic "sanity check" on all URLs in the found set. This can take
		a long time. Failed URLs are deactivated.
	*/
	function TestAllFoundLinks ( $http_vars )   
		{
		global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod, $user_prefix, $php_nuke_environment;

		set_time_limit(0);	// Make sure we don't time out.
		
		$results = "";
		if ( can_admin ( "EditLinks" ) )
			{
			$whereClause = "";
			
			if ( $http_vars[SEARCH_STRING] )
				{
				$whereClause = "WHERE ((_url LIKE '%".$http_vars[SEARCH_STRING]."%')";
				$whereClause .= " OR (_title LIKE '%".$http_vars[SEARCH_STRING]."%')";
				$whereClause .= " OR (_category LIKE '%".$http_vars[SEARCH_STRING]."%')";
				$whereClause .= " OR (_sub_category_1 LIKE '%".$http_vars[SEARCH_STRING]."%')";
				$whereClause .= " OR (_sub_category_2 LIKE '%".$http_vars[SEARCH_STRING]."%')";
				$whereClause .= " OR (_sub_category_3 LIKE '%".$http_vars[SEARCH_STRING]."%')";
				
				$sql = "SELECT _key FROM $c_CategoryTable WHERE _name LIKE '%".$http_vars[SEARCH_STRING]."%'";
		
				$result = sql_query ( $sql, $dbi );
				
				if ( $result )
					{
					while ( $row = sql_fetch_array ( $result, $dbi ) )
						{
						$whereClause .= " OR (_category='$row[_key]')";
						$whereClause .= " OR (_sub_category_1='$row[_key]')";
						$whereClause .= " OR (_sub_category_2='$row[_key]')";
						$whereClause .= " OR (_sub_category_3='$row[_key]')";
						}
					}
				
				if ( $php_nuke_environment )
					{
					$sql = "SELECT uname FROM $user_prefix"."_users WHERE (name LIKE '%".$http_vars[SEARCH_STRING]."%') ";
			
					$result = sql_query ( $sql, $dbi );
					
					if ( $result )
						{
						while ( list ( $uname ) = sql_fetch_array ( $result, $dbi ) )
							{
							$whereClause .= " OR (_user='$uname') OR (_user LIKE '%".$http_vars[SEARCH_STRING]."%')";
							}
						}
					}
				
				$whereClause .= " OR (_description LIKE '%".$http_vars[SEARCH_STRING]."%'))";
				}
			elseif ( $http_vars[FIND_CAT] )
				{
				if ( preg_match ( "/(.*)\%$/", $http_vars[FIND_CAT] ) )
					{
					$whereClause = "WHERE (_category LIKE '".$http_vars[FIND_CAT]."')\n";
					$whereclause .= " OR (_sub_category_1 LIKE '$http_vars[FIND_CAT]')\n";
					$whereclause .= " OR (_sub_category_2 LIKE '$http_vars[FIND_CAT]')\n";
					$whereclause .= " OR (_sub_category_3 LIKE '$http_vars[FIND_CAT]')\n";
					}
				else
					{
					if ( $http_vars[FIND_CAT] != "All_Categories" )
						{
						$whereClause = "WHERE (_category='$http_vars[FIND_CAT]')\n";
						$whereclause .= " OR (_sub_category_1='$http_vars[FIND_CAT]')\n";
						$whereclause .= " OR (_sub_category_2='$http_vars[FIND_CAT]')\n";
						$whereclause .= " OR (_sub_category_3='$http_vars[FIND_CAT]')\n";
						}
					}
				}
		
			$whereClause2 = "(_active='1')";
			if ( $whereClause )
				{
				$whereClause .= " AND $whereClause2";
				}
			else
				{
				$whereClause = "WHERE $whereClause2";
				}
		
			$sql = "SELECT * FROM $c_LinksTable $whereClause";
		
			$result = sql_query ( $sql, $dbi );
			
			if ( $result )
				{
				$linksTested = 0;
				$linksFailed = 0;
				while ( $row = sql_fetch_array ( $result, $dbi ) )
					{
					$retry = "";
					$deadLink = "";
					do
						{
						$test_result = intval ( Test_Link ( $row[_url] ) );
						$test_expl = "";
						if ( ($test_result >= 200) && ($test_result < 400) )
							{
							$test_expl = "";
							}
						elseif ( ($test_result >= 400) && ($test_result < 500) )
							{
							$test_expl = ""._ERROR_400."";
							$deadLink = 1;
							}
						elseif ( ($test_result >= 500) && ($test_result < 600) )
							{
							$test_expl = ""._ERROR_500."";
							$deadLink = 1;
							}
						elseif ( $test_result == 1 )
							{
							$test_expl = ""._ERROR_DNS."";
							$deadLink = 1;
							}
						elseif ( $test_result == 2 )
							{
							$test_expl = ""._ERROR_CONN."";
							$deadLink = 1;
							}
						else
							{
							$test_expl = ""._ERROR_UN."";
							$deadLink = 1;
							}
						
						if ( $deadLink )
							{
							if ( $retry )
								{
								$linksFailed++;
								$ur = htmlspecialchars ( $row[_url] );
								$results .= "<tr>\n<td colspan=\"2\">"._TEST_OF." &quot;$ur&quot; "._FAILED."</td>\n</tr>\n";
								$sql = "UPDATE $c_LinksTable SET _active='0' WHERE _url='$row[_url]'";
								$result2 = sql_query ( $sql, $dbi );
								if ( !$result2 )
									{
									$results .= "<tr>\n<td colspan=\"2\" style=\"text-align: center\"><h2>ERROR! UPDATE FAILED!</h2></td>\n</tr>\n";
									}
								$results .= "<tr>\n<td style=\"width:50px\">&nbsp;</td>\n<td>"._RETURN_VALUE." $test_result ($test_expl - "._LINK_DEACTIVATED.")</td>\n</tr>\n";
								$retry = "";
								}
							else
								{
								$retry = 1;
								}
							}
						else
							{
							$retry = "";
							}
						} while ( $retry );
					
					$linksTested++;
					}
				$results .= "<tr>\n<td colspan=\"2\" style=\"text-align: center\"><strong>$linksTested "._LINKS_WERE_TESTED.". $linksFailed "._LINKS_FAILED_TEST."</strong></td>\n</tr>\n";
				if ( $linksFailed )
					{
					$http_vars[SEARCH_STRING] = "";
					$http_vars[FIND_CAT] = "Inactive_Links";
					}
				}
			else
				{
				$results .= "<h2>"._ERROR._NO_LINKS_TESTED."</h2><br />\n";
				}
			}
		else
			{
			$results .= "<h2>"._INVALID."</h2><br />\n";
			}
		
		$http_vars[LINK_ACTION] = "";
		
		$http_vars[TEST_RESULTS] =  $results;
		}
	
	/***************************************************************************
		Tests one link. Called repeatedly from the previous function.
	*/
	function Test_Link ( $url )   
		{
		// 000 : only http://
		// 001 checkdnsrr error
		// 002 fsockopen error
		// other : http return code
		
		$parsed = parse_url ( $url );
		
		$host = $parsed["host"];
		$path = $parsed["path"];
		
		if ( !$path )
			{
			$path = "/";
			}
		
		if ( !checkdnsrr ( $host, "A" ) )
			{
			return "001";
			}
		
		$fp = fsockopen ( $host, 80, $errno, $errstr, 10 );
	
		if ( !$fp )
			return "002";
		
		fwrite ( $fp, "get $path HTTP/1.0\r\nAccept: */*\r\nHost: $host\r\nConnection: Keep-Alive\r\n\r\n" );  
		
		$line = fgets ( $fp, 1024 );  
			 
		if ( eregi ( "^http/[0-9]+.[0-9]+ ([0-9]{3}) [a-z ]*", $line) )
			{
			$return = trim ( eregi_replace ( "^http/[0-9]+.[0-9]+ ([0-9]{3}) [a-z ]*", "\\1", $line ) );
			}
		else
			{
			$return = "000";
			}
		
		fclose ( $fp );  
		
		return $return;
		}  
	}

if ( can_admin ( "EditCategories" ) )	// You have to be a category admin to get these functions
	{
	/***************************************************************************
		Presents a form that allows you to edit the categories
	*/
	function EditCategories ( $http_vars )
		{
		global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;
			
		if ( can_admin ( "EditCategories" ) && can_admin ( "EditLinks" ) )
			{
			print ( "<form action=\"$target_File\" method=\"$editormethod\"><div>\n" );
				print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
				print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
				print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
				print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
				print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
				print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
				print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
				print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
				print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"EditCategories\" />\n" );
				print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"\" />" );
				print ( "<table style=\"border:none\" cellpadding=\"3\" cellspacing=\"0\">\n" );
					print ( "<tr>\n" );
						print ( "<td  style=\"text-align: center\" colspan=\"2\"><hr /></td>\n" );
					print ( "</tr>\n" );
				
					$key = $http_vars[FIND_CAT];
					$name = "";
					$description = "";
					$key2 = "";
				
					if ( $key != "DudCat" )
						{
						print ( "<tr>\n" );
							if ( !$key )
								{
								print ( "<td  style=\"text-align: center\" colspan=\"2\"><h3><label for='CAT_SELECT'>"._EDIT_THE_CATEGORIES."</label></h3></td>\n" );
								}
							else
								{
								print ( "<td  style=\"text-align: center\" colspan=\"2\"><h3><label for='CAT_SELECT'>"._EDIT_THIS_CATEGORY."</label></h3></td>\n" );
								}
						print ( "</tr>\n" );
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: center\" colspan=\"2\"><hr /></td>\n" );
						print ( "</tr>\n" );
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: center\" colspan=\"2\">" );
								$sql = "SELECT * FROM $c_CategoryTable ORDER BY _name";
								$result = sql_query ( $sql, $dbi );
								
								if ( $result )
									{
									print ( "<select name=\"CAT_SELECT\" id=\"CAT_SELECT\" onchange=\"this.form.getElementById('FIND_CAT').value=this.value;this.form.submit()\">\n" );
										print ( "<option value=\"\"" );
										
										if ( !$key )
											{
											print ( " selected=\"selected\"" );
											}
										
										print ( ">Select A Category</option>\n" );
										while ( $row = sql_fetch_array ( $result, $dbi ) )
											{
											print ( "<option value=\"$row[_key]\"" );
											if ( $row[_key] == $key )
												{
												print ( " selected=\"selected\"" );
												}
											$nam = htmlspecialchars ($row[_name]);
											print ( ">$nam</option>\n" );
											}
										print ( "<option value=\"DudCat\">"._CREATE_A_NEW_CATEGORY."</option>\n" );
									print ( "</select>\n" );
									}
							print ( "</td>\n" );
						print ( "</tr>\n" );
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: center\" colspan=\"2\"><hr /></td>\n" );
						print ( "</tr>\n" );
						
						if ( $key )
							{
							$sql = "SELECT * FROM $c_CategoryTable WHERE _key='$key'";
							$result = sql_query ( $sql, $dbi );
							
							if ( $result )
								{
								$row = sql_fetch_array ( $result, $dbi );
								$key = $row[_key];
								$name = $row[_name];
								$description = $row[_description];
								}
							}
						$key2 = $key;
						}
					else
						{
						$key2 = "";
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: center\" colspan=\"2\"><h3>"._CREATE_A_NEW_CATEGORY."</h3></td>\n" );
						print ( "</tr>\n" );
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: center\" colspan=\"2\"><hr /></td>\n" );
						print ( "</tr>\n" );
						}
					
					if ( $key )
						{
						print ( "<tr>\n" );
							print ( "<td style=\"text-align: right\"><strong>Key:</strong></td>\n" );
							print ( "<td style=\"text-align:left\">" );
								if ( ($key == "unassigned_link") || ($key == "local") )
									{
									print ( $key2 );
									}
								else
									{
									print ( "<input type=\"text\" maxlength=\"127\" size=\"64\" name=\"CAT_KEY\" value=\"$key2\" />\n" );
									}
							print ( "</td>\n" );
						
						print ( "</tr>\n" );
						
						if ( preg_match ( "/(.*)\%$/", $key ) )
							{
							print ( "<tr>\n" );
								print ( "<td style=\"text-align: right\"><strong>NOTE:</strong></td>\n" );
								print ( "<td>"._CATCHALL."</td>\n" );
							print ( "</tr>\n" );
							}
						
						print ( "<tr>\n" );
							print ( "<td>&nbsp;</td>\n" );
							print ( "<td>"._CATCHALL_EXPLAIN."</td>\n" );
						print ( "</tr>\n" );
						
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: right\"><strong>"._NAME."</strong></td>\n" );
							print ( "<td>" );
								if ( ($key == "unassigned_link") || ($key == "local") )
									{
									print ( $name );
									}
								else
									{
									print ( "<input type=\"text\" maxlength=\"127\" size=\"64\" name=\"CAT_NAME\" value=\"$name\" />\n" );
									}
							print ( "</td>\n" );
						print ( "</tr>\n" );
						
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: right\" valign=\"top\"><strong>"._DESC."</strong></td>\n" );
							print ( "<td>" );
								if ( ($key == "unassigned_link") || ($key == "local") )
									{
									print ( $description );
									}
								else
									{
									print ( "<textarea maxchars=\"255\" cols=\"61\" rows=\"6\" name=\"CAT_DESC\" wrap=\"virtual\">$description</textarea>\n" );
									}
							print ( "</td>\n" );
						print ( "</tr>\n" );
						
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: center\" colspan=\"2\"><hr /></td>\n" );
						print ( "</tr>\n" );
	
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: center\" colspan=\"2\">" );
								print ( "<table style=\"border:none;width:100%\" cellpadding=\"5\" cellspacing=\"0\">\n" );
									print ( "<tr>\n" );
										print ( "<td  style=\"text-align: center;width:33%\">" );
											if ( $key == "DudCat" )
												{
												print ( "<input type=\"button\" value=\"Cancel\" onclick=\"this.form.submit()\" />" );
												}
											else
												{
												print ( "<input type=\"button\" value=\""._CANCEL."\" onclick=\"this.form.LINK_ACTION.value='';this.form.submit()\" />" );
												}
										print ( "</td>\n" );
										print ( "<td  style=\"text-align: center;width:34%\">" );
											if ( ($key == "unassigned_link") || ($key == "local") )
												{
												print ( "&nbsp;" );
												}
											else
												{
												print ( "<input type=\"reset\" value=\""._RESET_FORM."\" />" );
												}
										print ( "</td>\n" );
										print ( "<td  style=\"text-align: center;width:33%\">" );
											if ( $key == "DudCat" )
												{
												print ( "<input type=\"button\" value=\""._SUBMIT_FORM."\" onclick=\"this.form.LINK_ACTION.value='SaveCategory';this.form.submit()\" />\n" );
												}
											elseif ( ($key == "unassigned_link") || ($key == "local") )
												{
												print ( "&nbsp;" );
												}
											else
												{
												print ( "<input type=\"button\" value=\""._SUBMIT_FORM."\" onclick=\"this.form.FIND_CAT.value=this.form.CAT_SELECT.value;this.form.LINK_ACTION.value='SaveCategory';this.form.submit()\" />\n" );
												}
										print ( "</td>\n" );
									print ( "</tr>\n" );
								print ( "</table>" );
							print ( "</td>\n" );
						print ( "</tr>\n" );
						if ( ($key != "DudCat") && ($key != "unassigned_link") && ($key != "local") )
							{
							print ( "<tr>\n" );
								print ( "<td  style=\"text-align: center\" colspan=\"2\">" );
									print ( "<input type=\"button\" value=\""._DELETE_THIS_CATEGORY."\" onclick=\"if(confirm('"._DELETE_CATEGORY_CONFIRM."')){this.form.FIND_CAT.value=this.form.CAT_SELECT.value;this.form.LINK_ACTION.value='DeleteCategory';this.form.submit()}\" />" );
								print ( "</td>\n" );
							print ( "</tr>\n" );
							}
						}
					else
						{
						print ( "<tr>\n" );
							print ( "<td  style=\"text-align: center\" colspan=\"2\">" );
								print ( "<table style=\"border:none;width:100%\" cellpadding=\"5\" cellspacing=\"0\">\n" );
									print ( "<tr>\n" );
										print ( "<td  style=\"text-align: center\">" );
											print ( "<input type=\"button\" value=\""._CANCEL."\" onclick=\"this.form.LINK_ACTION.value='';this.form.submit()\" />" );
										print ( "</td>\n" );
									print ( "</tr>\n" );
								print ( "</table>" );
							print ( "</td>\n" );
						print ( "</tr>\n" );
						}
					
				print ( "</table>\n" );
			print ( "</div></form>" );
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		}
	
	/***************************************************************************
		Save an edited category, or create a new one
	*/
	function SaveCategory ( $http_vars )
		{
		global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;
			
		if ( can_admin ( "EditCategories" ) && can_admin ( "EditLinks" ) )
			{
			$find_cat = $http_vars[FIND_CAT];
			$cat_key = $http_vars[CAT_KEY];
			$cat_name = $http_vars[CAT_NAME];
			$cat_desc = htmlentities ( $http_vars[CAT_DESC] );
						
			if ( $cat_key && (($cat_key == $find_cat) || !CategoryMatch ( $http_vars )) )
				{
				if ( $cat_name )
					{
					$sql = "";
					
					if ( $find_cat )
						{
						$sql = "UPDATE $c_LinksTable SET _category='$cat_key' WHERE _category='$find_cat'";
						$result = sql_query ( $sql, $dbi );
						
						if ( $result )
							{
							print ( "<h3>Links Updated</h3>\n" );
							$sql = "UPDATE $c_CategoryTable SET _key='$cat_key',_name='$cat_name',_description='$cat_desc' WHERE _key='$find_cat'";
							$result = sql_query ( $sql, $dbi );
							
							if ( $result )
								{
								print ( "<h3>"._CATEGORY_UPDATED."</h3>" );
								}
							else
								{
								print ( "<h3>"._ERROR._CATEGORY_NOT_UPDATED."</h3>" );
								}
							}
						else
							{
							print ( "<h3>"._ERROR._CATEGORY_NOT_UPDATED."</h3>" );
							}
						}
					else
						{
						$sql = "INSERT INTO $c_CategoryTable (_key,_name,_description) VALUES ('$cat_key','$cat_name','$cat_desc')";
						$result = sql_query ( $sql, $dbi );
						
						if ( $result )
							{
							print ( "<h3>"._CATEGORY_ADDED."</h3>" );
							}
						else
							{
							print ( "<h3>"._ERROR._CATEGORY_NOT_ADDED."</h3>" );
							}
						}
					}
				else
					{
					print ( "<h3>"._ERROR._CATEGORY_NEEDS_NAME."</h3>" );
					}
				}
			else
				{
				if ( $cat_key )
					{
					print ( "<h3>"._ERROR."\"$http_vars[CAT_KEY]\" "._ALREADY_EXISTS."</h3>" );
					}
				else
					{
					print ( "<h3>"._ERROR._CATEGORY_NEEDS_KEY."</h3>" );
					}
				}
			
			EditCategories ( $http_vars );
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		}
	
	/***************************************************************************
		Returns true if the category exists
	*/
	function CategoryMatch ( $http_vars )
		{
		global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;
	
		$ret = "";
	
		if ( can_admin ( "EditCategories" ) && can_admin ( "EditLinks" ) )
			{
			$sql = "SELECT * FROM $c_CategoryTable WHERE _key='$http_vars[CAT_KEY]'";
			$result = sql_query ( $sql, $dbi );
			
			if ( $result )
				{
				$row = sql_fetch_array ( $result, $dbi );
				if ( $row )
					{
					$ret = 1;
					}
				}
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		
		return $ret;
		}
	
	/***************************************************************************
		Delete the given category
	*/
	function DeleteCategory ( $http_vars )
		{
		global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;
	
		if ( can_admin ( "EditCategories" ) && can_admin ( "EditLinks" ) )
			{
			$sql = "UPDATE $c_LinksTable SET _category='unassigned_link' WHERE _category='$http_vars[FIND_CAT]'";
			$result = sql_query ( $sql, $dbi );
			
			if ( $result )
				{
				print ( "<h3>"._LINKS_UNASSIGNED."</h3>\n" );
				$sql = "DELETE FROM $c_CategoryTable WHERE _key='$http_vars[FIND_CAT]'";
				$result = sql_query ( $sql, $dbi );
				if ( $result )
					{
					print ( "<h3>"._CATEGORY_DELETED."</h3>" );
					}
				else
					{
					print ( "<h3>"._ERROR._CATEGORY_NOT_DELETED."</h3>" );
					}
		
				EditCategories ( $http_vars );
				}
			else
				{
				print ( "<h3>"._ERROR._CATEGORY_NOT_DELETED."</h3>" );
				}
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		}
	
	/***************************************************************************
		Check to see if the links table exists
	*/
	function CheckForLinkTable ( )
		{
		$ret = 1;
		
		global $c_LinksTable, $dbi;
		
		$sql = "SELECT * FROM $c_LinksTable";
		$result = sql_query ( $sql, $dbi );
		
//		echo "CheckForLinkTable(): \$sql = \"$sql\", \$result = \"$result\"<br />\n";

		if ( !$result )
			{
			$ret = 0;
			}
		
		return $ret;
		}
	
	/***************************************************************************
		Check to see if the category table exists
	*/
	function CheckForCatTable ( )
		{
		$ret = 1;
		
		global $c_CategoryTable, $dbi;
		
		$sql = "SELECT * FROM $c_CategoryTable";
		
		$result = sql_query ( $sql, $dbi );
		
//		echo "CheckForCatTable(): \$sql = \"$sql\", \$result = \"$result\"<br />\n";
		
		if ( !$result )
			{
			$ret = 0;
			}
		
		return $ret;
		}
		
	/***************************************************************************
		Check for the tables, and create them if they don't exist
	*/
	function Check_For_And_CreateTables ( )
		{
		global $c_LinksTable, $c_CategoryTable, $dbi;
		
		if ( !CheckForCatTable ( ) )
			{
			$sql = "CREATE TABLE $c_CategoryTable (
					  _key varchar(127) NOT NULL default 'EMPTY',
					  _name varchar(255) NOT NULL default 'NEEDS A DESCRIPTION',
					  _description mediumtext,
					  PRIMARY KEY  (_key),
					  KEY _name (_name)
					) type=MyISAM;";
	
			$result = sql_query ( $sql, $dbi );
			
//			echo "Check_For_And_CreateTables() -Category Table: \$sql = \"$sql\", \$result = \"$result\"<br />\nSQL Error = \"".sql_error()."\"";
					
			$sql = "INSERT INTO $c_CategoryTable VALUES ('local', 'Pages on this Web site', 'This type of link leads to another page on this Web site.');";
	
			$result = sql_query ( $sql, $dbi );
			
	//		echo "Check_For_And_CreateTables() -Category Table: \$sql = \"$sql\", \$result = \"$result\"<br />\nSQL Error = \"".sql_error()."\"";
			}
		
		if ( !CheckForLinkTable ( ) )
			{
			$sql = "CREATE TABLE $c_LinksTable (
					  _url varchar(255) NOT NULL default 'REPLACE WITH NEW URL',
					  _title varchar(255) NOT NULL default '',
					  _description text NOT NULL,
					  _category varchar(127) NOT NULL default '',
					  _timestamp timestamp(14) NOT NULL,
					  _user varchar(255) NOT NULL default '',
					  _active tinyint(4) NOT NULL default '0',
					  _sub_category_1 varchar(127) NOT NULL default '',
					  _sub_category_2 varchar(127) NOT NULL default '',
					  _sub_category_3 varchar(127) NOT NULL default '',
					  PRIMARY KEY  (_url),
					  KEY _title (_title),
					  KEY _category (_category),
					  KEY _timestamp (_timestamp),
					  KEY _sub_category_1 (_sub_category_1),
					  KEY _sub_category_2 (_sub_category_2),
					  KEY _sub_category_3 (_sub_category_3)
					) type=MyISAM;";
	
			$result = sql_query ( $sql, $dbi );
			
	//		echo "Check_For_And_CreateTables() -Links Table: \$sql = \"$sql\", \$result = \"$result\"<br />\nSQL Error = \"".sql_error()."\"";
			}
		}
	}

/***************************************************************************
	Normal functions -any user can get these
*/

/***************************************************************************
	Deletes a single Link
*/
function DeleteLink ( $http_vars )
	{
	global $moderate_links, $user, $userinfo, $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;

	$sql = "SELECT * FROM $c_LinksTable WHERE _url='$http_vars[LINKURL]'";
	$result = sql_query ( $sql, $dbi );
	if ( $result )
		{
		$row = sql_fetch_array ( $result, $dbi );
		$usereditor = "";
		$userinfo = getusrinfo ( $user );
		if ( $php_nuke_environment && !$moderate_links && ($userinfo[uname] == $row[_user]) )
			{
			$usereditor = 1;
			}
		
		if ( can_admin ( "EditLinks" ) || $usereditor )
			{
			$sql = "DELETE FROM $c_LinksTable WHERE _url='$http_vars[LINKURL]'";
			$result = sql_query ( $sql, $dbi );
			print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none;width:100%\">\n" );
				print ( "<tr>\n" );
					print ( "<td  style=\"text-align: center\">" );
					if ( $result )
						{
						print ( "<h3>"._LINK_DELETED."</h3>" );
						}
					else
						{
						print ( "<h3>"._ERROR._LINK_NOT_DELETED."</h3>" );
						}
					print ( "</td>\n" );
				print ( "</tr>\n" );
			print ( "</table>\n" );
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		}
	DisplayLinks ( $http_vars );
	}

/***************************************************************************
	Retruns true if the URL already exists
*/
function HasMatch ( $http_vars )
	{
	global $c_LinksTable, $dbi;

	$ret = "";

	$sql = "SELECT * FROM $c_LinksTable WHERE _url='$http_vars[NEWURL]'";
	$result = sql_query ( $sql, $dbi );
	
	if ( $result )
		{
		$row = sql_fetch_array ( $result, $dbi );
		if ( $row )
			{
			$ret = $row[_title];
			}
		}
	
	return $ret;
	}

/***************************************************************************
	This checks to see if the URL already exists
*/
function VerifyURL ( $http_vars )
	{
	global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod;

	$match = HasMatch ( $http_vars );
	if ( $match )
		{
		print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none;width:100%\">\n" );
			print ( "<tr>\n" );
				print ( "<td  style=\"text-align: center\">" );
					print ( "<h3>"._ERROR._LINK_NOT_UNIQUE."</h3>" );
				print ( "</td>\n" );
			print ( "</tr>\n" );
			print ( "<tr>\n" );
				print ( "<td style=\"text-align:left\">" );
					print ( "<h4>"._EXPLAIN_NOT_UNIQUE."</h4>" );
				print ( "</td>\n" );
			print ( "</tr>\n" );
			print ( "<tr>\n" );
				print ( "<td style=\"text-align:left\">" );
					print ( "<h4>"._CLICK_BACK."</h4>" );
				print ( "</td>\n" );
			print ( "</tr>\n" );
		print ( "</table>\n" );
		$http_vars[LINKURL] = $http_vars[NEWURL];
		PrintOneLinkToEdit ( $http_vars );
		}
	}

/***************************************************************************
	Submits an edited link
*/
function SubmitLink ( $http_vars )
	{
	global $moderate_links, $user, $userinfo, $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod, $default_category, $php_nuke_environment;

	$sql = "SELECT * FROM $c_LinksTable WHERE _url='$http_vars[LINKURL]'";
	$result = sql_query ( $sql, $dbi );
	if ( $result )
		{
		$row = sql_fetch_array ( $result, $dbi );
		$usereditor = "";

		$userinfo = getusrinfo ( $user );
		if ( $php_nuke_environment && !$moderate_links && ($userinfo[uname] == $row[_user]) )
			{
			$usereditor = 1;
			}
		
		if ( $http_vars[NEWURL] && $http_vars[TITLE] )
			{
			if ( can_admin ( "EditLinks" ) || $usereditor )
				{
				$column = "";
				
				if ( $http_vars[NEWURL] != $http_vars[LINKURL] )
					{
					if ( !HasMatch ( $http_vars ) )
						{
						$column = "_url='$http_vars[NEWURL]',";
						}
					else
						{
						VerifyURL ( $http_vars );
						return;
						}
					}
				
				$encodedTitle = htmlentities ( $http_vars[TITLE] );
				$encodedDescription = htmlentities ( $http_vars[DESCRIPTION] );
				
				if ( $http_vars[ACTIVE_LINK] )
					{
					$http_vars[ACTIVE_LINK] = '1';
					}
				else
					{
					$http_vars[ACTIVE_LINK] = '0';
					}
				
				$column .= "_active='$http_vars[ACTIVE_LINK]',_category='$http_vars[FIND_CAT]',_sub_category_1='$http_vars[_sub_category_1]',_sub_category_2='$http_vars[_sub_category_2]',_sub_category_3='$http_vars[_sub_category_3]',_title='$encodedTitle',_description='$encodedDescription'";
				
				// Wildcard categories aren't allowed in submitted links.
				if ( strpos ( "%", $http_vars['FIND_CAT'] ) )
					{
					$http_vars['FIND_CAT'] = '';
					}
				
				$sql = "UPDATE $c_LinksTable SET $column WHERE _url='$http_vars[LINKURL]'";
				// Make sure we have a valid URL, category and title (Description is optional)
				if ( $http_vars['FIND_CAT'] && $http_vars['NEWURL'] && $encodedTitle )
					{
					$result = sql_query ( $sql, $dbi );
					}
				else
					{
					$result = '';
					}
				
				if ( $result )
					{
					print ( "<h2>"._LINK_SUBMITTED."</h2>" );
					}
				else
					{
					print ( "<h2>"._ERROR._LINK_NOT_SUBMITTED."</h2>" );
					}
				
				return 1;
				}
			}
		}

	print ( "<div style=\"text-align: center\"><h4>"._LINK_NOT_SUBMITTED."</h4><br /><a href=\"javascript:history.back()\">"._GO_BACK."</a></div>" );
	
	return "";
	}

/***************************************************************************
	Presents the selected=\"selected\" link in an editable form
*/
function PrintOneLinkToEdit ( $http_vars )
	{
	global $instructions_url, $moderate_links, $user, $userinfo, $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $editormethod, $usermethod, $user_prefix, $php_nuke_environment;

	$sql = "SELECT * FROM $c_LinksTable WHERE _url='$http_vars[LINKURL]'";
	$result = sql_query ( $sql, $dbi );
	if ( $result )
		{
		$row = sql_fetch_array ( $result, $dbi );
		$usereditor = "";

		$userinfo = getusrinfo ( $user );
		if ( $php_nuke_environment && !$moderate_links && ($userinfo[uname] == $row[_user]) )
			{
			$usereditor = 1;
			}

		if ( can_admin ( "EditLinks" ) || $usereditor )
			{
			$sql = "SELECT * FROM $c_LinksTable WHERE _url='$http_vars[LINKURL]'";
			$result = sql_query ( $sql, $dbi );
			
			if ( $result )
				{
				$row = sql_fetch_array ( $result, $dbi );
				if ( $row )
					{
					$title = $row[_title];
					$url = $row[_url];
					$description = $row[_description];
					
					$url = htmlentities ( $url );
						print ( "<script type=\"text/javascript\">\n<!--\nvar g_LinkTested=0;\n//--></script>\n" );
						print ( "<div style=\"text-align: center\">\n<form id=\"LinkEditor\" action=\"$target_File\"" );
							if ( can_admin ( "EditLinks" ) )
								{
								print ( " method=\"$editormethod\"" );
								}
							else
								{
								print ( " onSubmit=\"if(this.LINK_ACTION.value && !g_LinkTested){alert('"._MUST_TEST_LINK."');return false;}else return true;\"" );
								print ( " method=\"$usermethod\"" );
								}
						
						print ( "><div>\n" );
						print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
						print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
						print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
						print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
						print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
						print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"Timestamp\" />\n" );
						print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
						print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
						print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
						print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"Submit\" />\n" );
						print ( "<input type=\"hidden\" name=\"LINKURL\" value=\"$http_vars[LINKURL]\" />\n" );
						print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none\">\n" );
							if ( !can_admin ( "EditLinks" ) )
								{
								print ( "<tr>\n<td  colspan=\"2\" style=\"text-align: center\">" );
									if ( $instructions_url )	// Instruction link is placed to the right of the title
										{
										print ( "<table  style=\"border:none;width:100%\">\n<tr>\n<td style=\"text-align: center\">" );
										}
									print ( "<h2>"._EDIT_A_LINK."</h2>\n" );
									if ( $instructions_url )
										{
										print ( "</td><td  style=\"text-align: right\"><a href=\"".htmlentities ( $instructions_url )."\">".htmlentities ( _INSTRUCTIONS )."</a></td>\n" );
										print ( "</tr></table>\n" );
										}
								print ( "</td></tr>\n" );
								}
							print ( "<tr>\n" );
								print ( "<td  style=\"text-align: right;font-weight:bold\"><label for='NEWURL'>"._URL."</label></td>\n" );
								print ( "<td style=\"text-align:left\"><input type=\"text\" maxlength=\"255\" size=\"64\" id=\"NEWURL\" name=\"NEWURL\" value=\"$url\" /></td>\n" );
							print ( "</tr>\n" );
							print ( "<tr>\n" );
								print ( "<td  style=\"text-align: right;font-weight:bold\"><label for='TITLE'>"._NAME."</label></td>\n" );
								print ( "<td style=\"text-align:left\"><input type=\"text\" maxlength=\"255\" size=\"64\" id=\"TITLE\" name=\"TITLE\" value=\"$title\" /></td>\n" );
							print ( "</tr>\n" );
							print ( "<tr>\n" );
								print ( "<td  style=\"text-align: right;font-weight:bold\"><label for='FIND_CAT'>"._MAIN_CATEGORY."</label></td>\n" );
								print ( "<td style=\"text-align:left\">" );
									PrintCategoryPopup ( $row[_category], 1, 1, $http_vars );
								print ( "</td>\n" );
							print ( "</tr>\n" );
							print ( "<tr>\n" );
								print ( "<td  style=\"text-align: right;font-weight:bold\"><label for='_sub_category_1'>"._SUB_CATEGORY_1."</label></td>\n" );
								print ( "<td style=\"text-align:left\">" );
									PrintCategoryPopup ( $row[_sub_category_1], 1, 1, $http_vars, '_sub_category_1' );
								print ( "</td>\n" );
							print ( "</tr>\n" );
							print ( "<tr>\n" );
								print ( "<td  style=\"text-align: right;font-weight:bold\"><label for='_sub_category_2'>"._SUB_CATEGORY_2."</label></td>\n" );
								print ( "<td style=\"text-align:left\">" );
									PrintCategoryPopup ( $row[_sub_category_2], 1, 1, $http_vars, '_sub_category_2' );
								print ( "</td>\n" );
							print ( "</tr>\n" );
							print ( "<tr>\n" );
								print ( "<td  style=\"text-align: right;font-weight:bold\"><label for='_sub_category_3'>"._SUB_CATEGORY_3."</label></td>\n" );
								print ( "<td style=\"text-align:left\">" );
									PrintCategoryPopup ( $row[_sub_category_3], 1, 1, $http_vars, '_sub_category_3' );
								print ( "</td>\n" );
							print ( "</tr>\n" );
							print ( "<tr>\n" );
								print ( "<td  valign=\"top\" style=\"text-align: right;font-weight:bold\"><label for='DESCRIPTION'>"._DESC."</label></td>\n" );
								print ( "<td style=\"text-align:left\"><textarea maxchars=\"255\" rows=\"10\" cols=\"61\" name=\"DESCRIPTION\" id=\"DESCRIPTION\">$description</textarea></td>\n" );
							print ( "</tr>\n" );
							if ( can_admin ( "EditLinks" ) )
								{
								print ( "<tr>\n" );
									print ( "<td  valign=\"top\" style=\"text-align: right;font-weight:bold\"><label for='ACTIVE_LINK'>"._ACTIVE."</label></td>\n" );
									print ( "<td style=\"text-align:left\"><input type=\"CheckBox\" value=\"1\" id=\"ACTIVE_LINK\" name=\"ACTIVE_LINK\"" );
									if ( $row[_active] )
										{
										print ( " checked=\"checked\"" );
										}
									print ( "></td>\n" );
								print ( "</tr>\n" );
								}
							else
								{
								if ( !$moderate_links )
									{
									print ( "<input type=\"hidden\" name=\"ACTIVE_LINK\" value=\"1\" />\n" );
									}
								}
							if ( $php_nuke_environment && can_admin ( "EditLinks" ) )
								{
								print ( "<tr>\n" );
									$uname = $row[_user];
									$sql_u = "SELECT name FROM $user_prefix"."_users where uname='$uname'";
									$uresult = sql_query ( $sql_u, $dbi );
									
									if ( !$uresult )
										{
										$sql_u = "SELECT name FROM $user_prefix"."_users where username='$uname'";
										$uresult = sql_query ( $sql_u, $dbi );
										}
									
									if ( $uresult )
										{
										list ( $name ) = sql_fetch_array ( $uresult, $dbi);
										
										if ( !$name )
											{
											$name = $uname;
											}
										}
									
									print ( "<td  style=\"text-align: right;font-weight:bold\">"._SUBMITTED_BY."</td>\n" );
									print ( "<td style=\"text-align:left\"><a href=\"modules.php?name=Your_Account&amp;op=userinfo&amp;uname=$uname\">$name</a></td>\n" );
								print ( "</tr>\n" );
								}
							print ( "<tr>\n" );
								print ( "<td  colspan=\"2\" style=\"text-align: center;font-weight:bold\">" );
									print ( "<a href=\"javascript:TestURL()\">"._TEST_THIS_URL._OPENS_NEW_WINDOW."</a>" );
									print ( "<script type=\"text/javascript\">\n<!-- // Hide from old browsers\n" );
										print ( "function TestURL(){\n" );
											print ( "g_LinkTested = 1;\n" );
											print ( "var url = document.getElementById('LinkEditor').NEWURL.value;\n" );
											print ( "window.open(url,'test','location=1');\n" );
										print ( "}\n" );
									print ( "// Done Hiding -->\n</script>" );
								print ( "</td>\n" );
							print ( "</tr>\n" );
							print ( "<tr>\n" );
								print ( "<td  colspan=\"2\" style=\"text-align: center\"><table style=\"border:none;width:100%\" cellpadding=\"0\" cellspacing=\"0\">\n" );
									print ( "<tr>\n" );
										print ( "<td style=\"text-align: center;width:33%\"><input type=\"submit\" value=\""._DELETE_THIS_LINK."\" onclick=\"if(confirm('"._DELETE_LINK_CONFIRM."')){this.form.LINK_ACTION.value='Delete';return true;}else return false\"></td />\n" );
										print ( "<td style=\"text-align: center;width:34%\"><input type=\"submit\" value=\""._CANCEL."\" onclick=\"this.form.LINK_ACTION.value=''\"></td />\n" );
										print ( "<td style=\"text-align: center;width:33%\"><input type=\"submit\" value=\""._SUBMIT_THIS_LINK."\"></td />\n" );
									print ( "</tr>\n" );
								print ( "</table></td>\n" );
							print ( "</tr>\n" );
						print ( "</table>\n" );
					print ( "</div></form></div>" );
					}
				}
			}
		else
			{
			print ( "<h2>"._INVALID."</h2><br />\n" );
			}
		}
}

/***************************************************************************
	Print the category selection popup menu
*/
function PrintCategoryPopup ( $initCat, $editor, $edit_link, $http_vars, $inName = "FIND_CAT", $changeAction = "" )
	{
	global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi;
	
	$sql = "SELECT * FROM $c_CategoryTable ORDER BY _name";
	$result = sql_query ( $sql, $dbi );
	
	if ( $result )
		{
		if ( !$inName )
			{
			$inName = "FIND_CAT";
			}
		
		print ( "<select id=\"$inName\" name=\"$inName\"" );
		
		if ( !$editor && !$edit_link )
			{
			print ( " title=\""._SELECT_A_CATEGORY_TITLE."\"" );
			}

		if ( !$edit_link )
			{
			print ( " onchange=\"" );
			if ( $changeAction )
				{
				print ( "$changeAction; " );
				}
			print ( "this.form.submit()" );
			print ( "\"" );
			}
			
		print ( ">\n" );
			
		if ( !$edit_link )
			{
			print ( "<option value=\"All_Categories\"" );
			if ( !$initCat || ($initCat == "All_Categories") )
				{
				print ( " selected=\"selected\"" );
				}
			print ( ">"._DISPLAY_EVERY_CATEGORY."</option>\n" );
			
			if ( can_admin ( "EditLinks" ) )
				{
				print ( "<option value=\"Inactive_Links\"" );
				if ( $initCat == "Inactive_Links" )
					{
					print ( " selected=\"selected\"" );
					}
				print ( ">"._INACTIVE_LINKS."</option>\n" );
				}
			}
		else
			{
			print ( "<option value=\"\"" );
			if ( !$initCat )
				{
				print ( " selected=\"selected\"" );
				}
			print ( ">"._SELECT_A_CATEGORY."</option>\n" );
			}
		
		while ( $row = sql_fetch_array ( $result, $dbi ) )
			{
			if ( $editor || $edit_link || ($row[_key] != "local" ) )
				{
				if ( !($edit_link && preg_match ( "/(.*)\%$/", $row[_key] )) )
					{
					print ( "<option value=\"$row[_key]\"" );
					if ( $row[_key] == $initCat )
						{
						print ( " selected=\"selected\"" );
						}
					$name = htmlspecialchars ($row[_name]);
					print ( ">$name</option>\n" );
					}
				}
			}
		print ( "</select>\n" );
		}
	else
		print "ERROR! \$sql = &quot;$sql&quot;<br />SQL Error = &quot;".sql_error()."&quot";
	}

/***************************************************************************
	Presents the Add Link form
*/

function PrintOneLinkToAdd ( $http_vars )
	{
	global $c_LinksTable, $editormethod, $usermethod, $c_CategoryTable, $target_File, $mod_Name, $dbi, $user, $moderate_links, $instructions_url;
	
	if ( is_user ( $user ) )
		{
		$title = "ADD TITLE HERE";
		$url = "ADD SITE URL HERE";
		
		print ( "<script type=\"text/javascript\">\n<!--\nvar g_LinkTested=0;\n//--></script>\n" );
		print ( "<div style=\"text-align: center\">\n<form id=\"LinkEditor\" action=\"$target_File\"" );
			if ( can_admin ( "EditLinks" ) )
				{
				print ( " method=\"$editormethod\"" );
				}
			else
				{
				print ( " onSubmit=\"if(this.LINK_ACTION.value && !g_LinkTested){alert('"._MUST_TEST_LINK."');return false;}else return true;\"" );
				print ( " method=\"$usermethod\"" );
				}
			
			print ( "><div>\n" );
			print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
			print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
			print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
			print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
			print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
			print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"Timestamp\" />\n" );
			print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
			print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
			print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
			print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"\" />\n" );
			print ( "<input type=\"hidden\" name=\"LINKURL\" value=\"$http_vars[LINKURL]\" />\n" );
			print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none\">\n" );
				print ( "<tr>\n<td  colspan=\"2\" style=\"text-align: center\">" );
					if ( $instructions_url )	// Instruction link is placed to the right of the title
						{
						print ( "<table  style=\"border:none;width:100%\">\n<tr>\n<td style=\"text-align: center\">" );
						}
					print ( "<h2>"._SUBMIT_A_NEW_LINK."</h2>\n" );
					if ( $instructions_url )
						{
						print ( "</td><td  style=\"text-align: right\"><a href=\"".htmlentities ( $instructions_url )."\">".htmlentities ( _INSTRUCTIONS )."</a></td>\n" );
						print ( "</tr></table>\n" );
						}
				print ( "</td></tr>\n" );
				print ( "<tr>\n" );
					print ( "<td  style=\"text-align: right;font-weight:bold\"><label for=\"NEWURL\">"._URL."</label></td>\n" );
					print ( "<td style=\"text-align:left\"><input type=\"text\" maxlength=\"255\" size=\"64\" id=\"NEWURL\" name=\"NEWURL\" value=\"$url\" /></td>\n" );
				print ( "</tr>\n" );
				print ( "<tr>\n" );
					print ( "<td  style=\"text-align: right;font-weight:bold\"><label for=\"TITLE\">"._NAME."</label></td>\n" );
					print ( "<td style=\"text-align:left\"><input type=\"text\" maxlength=\"255\" size=\"64\" id=\"TITLE\" name=\"TITLE\" value=\"$title\"></td />\n" );
				print ( "</tr>\n" );
				print ( "<tr>\n" );
					print ( "<td  style=\"text-align: right;font-weight:bold\"><label for=\"FIND_CAT\"><label for=\"FIND_CAT\">"._CATEGORY."</label></label></td>\n" );
					print ( "<td style=\"text-align:left\">" );
						$initCat = $http_vars[FIND_CAT];
						if ( preg_match ( "/(.*)\%$/", $initCat ) )
							{
							$sql = "SELECT _key FROM $c_CategoryTable WHERE (_key LIKE '$initCat') AND (_key<>'$initCat') ORDER BY _name";
							$initCat = "";
							
							$res = sql_query ( $sql, $dbi );
							
							if ( $res )
								{
								$numrows = sql_num_rows ( $res, $dbi );
								list ( $initCat ) = sql_fetch_array ( $res, $dbi );
								}
							}
						PrintCategoryPopup ( $initCat, 0, 1, $http_vars );
					print ( "</td>\n" );
				print ( "</tr>\n" );
				print ( "<tr>\n" );
					print ( "<td  valign=\"top\" style=\"text-align: right;font-weight:bold\"><label for=\"DESCRIPTION\">"._DESC."</label></td>\n" );
					print ( "<td style=\"text-align:left\"><textarea maxchars=\"512\" rows=\"10\" cols=\"61\" name=\"DESCRIPTION\" id=\"DESCRIPTION\"></textarea></td>\n" );
				print ( "</tr>\n" );
				if ( can_admin ( "EditLinks" ) )
					{
					print ( "<tr>\n" );
						print ( "<td  valign=\"top\" style=\"text-align: right;font-weight:bold\"><label for=\"ACTIVE_LINK\">"._ACTIVE."</label></td>\n" );
						print ( "<td style=\"text-align:left\"><input type=\"CheckBox\" value=\"1\" id=\"ACTIVE_LINK\" name=\"ACTIVE_LINK\"" );
						if ( !$moderate_links )
							{
							print ( " checked=\"checked\"" );
							}
						print ( "></td>\n" );
					print ( "</tr>\n" );
					}
				print ( "<tr>\n" );
					print ( "<td  colspan=\"2\" style=\"text-align: center\"><a href=\"javascript:TestURL()\"><strong>"._TEST_THIS_URL._OPENS_NEW_WINDOW."</strong></a>" );
						print ( "<script type=\"text/javascript\">\n<!-- // Hide from old browsers\n" );
							print ( "function TestURL(){\n" );
								print ( "g_LinkTested = 1;\n" );
								print ( "var url = document.getElementById('LinkEditor').NEWURL.value;\n" );
								print ( "window.open(url,'test','location=1');\n" );
							print ( "}\n" );
						print ( "// Done Hiding -->\n</script></td>\n" );
				print ( "</tr>\n" );
				print ( "<tr>\n" );
					print ( "<td  colspan=\"2\" style=\"text-align: center\"><table style=\"border:none;width:100%\" cellpadding=\"0\" cellspacing=\"0\">\n" );
						print ( "<tr>\n" );
							print ( "<td style=\"text-align: center;width:33%\"><input type=\"submit\" value=\""._CANCEL."\"></td />\n" );
							print ( "<td style=\"text-align: center;width:34%\"><input type=\"reset\" value=\""._RESET_FORM."\"></td />\n" );
							print ( "<td style=\"text-align: center;width:33%\"><input type=\"submit\" value=\""._SUBMIT_FORM."\" onclick=\"this.form.LINK_ACTION.value='AddNewLink'\"></td />\n" );
						print ( "</tr>\n" );
					print ( "</table></td>\n" );
				print ( "</tr>\n" );
			print ( "</table>\n" );
		print ( "</div></form>\n</div>\n" );
		}
	}

/***************************************************************************
	Add a new link from the Add a New Link form
*/
function AddNewLink ( $http_vars )
	{
	global $c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $user, $userinfo, $moderate_links;

	if ( is_user ( $user ) )
		{
		$userinfo = getusrinfo ( $user );
		
		$sql = "SELECT * FROM $c_LinksTable WHERE _url='$http_vars[NEWURL]'";
		
		$result = sql_query ( $sql, $dbi );
		
		if ( $result && sql_num_rows ( $result, $dbi ) )
			{
			$alreadyExists = 1;
			}
		
		if ( $alreadyExists )
			{
			print ( "<div style=\"text-align: center\"><h4>"._THIS_URL." (".htmlspecialchars ( $http_vars[NEWURL] ).") "._ALREADY_EXISTS."</h4><br /><a href=\"javascript:history.back()\">"._GO_BACK."</a></div>" );
			return "";
			}
		else
			{
			if ( !$moderate_links )
				{
				$http_vars[ACTIVE_LINK] = 1;
				}
			
			$sql = "INSERT INTO $c_LinksTable (_url,_title,_description,_category,_user,_active) VALUES ('$http_vars[NEWURL]','$http_vars[TITLE]','$http_vars[DESCRIPTION]','$http_vars[FIND_CAT]','$userinfo[uname]','$http_vars[ACTIVE_LINK]')";
			$result = sql_query ( $sql, $dbi );
			print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none;width:100%\">\n" );
				print ( "<tr>\n" );
					print ( "<td  style=\"text-align: center\">" );
					if ( $result )
						{
						print ( "<h3>"._LINK_SUBMITTED."</h3>" );
						}
					else
						{
						print ( "<h3>"._ERROR._LINK_NOT_SUBMITTED."</h3>" );
						}
					print ( "</td>\n" );
				print ( "</tr>\n" );
				if ( $result && !can_admin ( "EditLinks" ) && $moderate_links )
					{
					print ( "<tr>\n" );
						print ( "<td style=\"text-align:left\">" );
								print ( ""._WILL_REVIEW."" );
						print ( "</td>\n" );
					print ( "</tr>\n" );
					}
			print ( "</table>\n" );
			return 1;
			}
		}
	}

/***************************************************************************
	Main link display function
*/

function DisplayLinks ( $http_vars )
	{
	global	$c_LinksTable, $c_CategoryTable, $target_File, $mod_Name, $dbi, $php_nuke_environment;
	global	$user_prefix, $user, $userinfo, $editormethod, $usermethod;
	global	$default_category, $page_threshold, $default_sort, $moderate_links, $replace_URL_Header;
	global	$dbhost, $dbuname, $dbpass, $dbname;

	$dbi = sql_connect( $dbhost, $dbuname, $dbpass, $dbname );
		
	$editor = "";
	
	// The main administrator can install the tables if they don't exist
	if ( can_admin ( "EditLinks" ) && can_admin ( "EditCategories" ) )
		{
		Check_For_And_CreateTables ( );
		}
	
	if ( can_admin ( "EditLinks" ) || can_admin ( "EditCategories" ) )
		{
		$editor = 1;
		}
	
	// We set the category to the default if no previous one specified.
	if ( !$http_vars[SEARCH_STRING] && !$http_vars[FIND_CAT] && $default_category )
		{
		// We verify that the default category exists
		$sql = "SELECT _key FROM $c_CategoryTable WHERE _key='$default_category'";

		$result = sql_query ( $sql, $dbi );
		
		if ( $result )
			{
			list ( $default_category ) = sql_fetch_array ( $result, $dbi );
			}
		else
			{
			$default_category = "";
			}
			
		$http_vars[FIND_CAT] = $default_category;
		}
	
	$startCount = $http_vars[START_COUNT];
	
	if ( $http_vars[DISP_COUNT] )
		{
		$startCount = intval ($startCount / $http_vars[DISP_COUNT]) * $http_vars[DISP_COUNT];
		}
	
	$endCount = $startCount + $http_vars[DISP_COUNT];
	
	if (($http_vars[DISP_COUNT] == -1) || ($endCount <= $startCount) )
		{
		$endCount = 0;
		}
	
	$first_Sort = "_timestamp DESC";
	$second_Sort = "_category";
	$third_Sort = "_title";
	$whereClause = "";
	
	if ( $http_vars[SORT_BY] == "Title" )
		{
		$first_Sort = "_title";
		$second_Sort = "_category";
		$third_Sort = "_timestamp DESC";
		}
	
	if ( $http_vars[SORT_BY] == "Category_Title" )
		{
		$first_Sort = "_category";
		$second_Sort = "_title";
		$third_Sort = "_timestamp DESC";
		}
	
	if ( $http_vars[SORT_BY] == "Category_Time" )
		{
		$first_Sort = "_category";
		$second_Sort = "_timestamp DESC";
		$third_Sort = "_title";
		}
	
	if ( $php_nuke_environment )
		{
		if ( $http_vars[SORT_BY] == "Submitter" )
			{
			$first_Sort = "_user";
			if ( preg_match ( "/(.*)\%$/", $http_vars[FIND_CAT] ) )
				{
				$second_Sort = "_timestamp DESC";
				$third_Sort = "_category";
				}
			else
				{
				$second_Sort = "_timestamp DESC";
				$third_Sort = "_title";
				}
			}
		}
	
	$whereClause = "";
	
	if ( $http_vars[SEARCH_STRING] )
		{
		$whereClause = "WHERE ((_url LIKE '%".$http_vars[SEARCH_STRING]."%')";
		$whereClause .= " OR (_title LIKE '%".$http_vars[SEARCH_STRING]."%')";
		$whereClause .= " OR (_category LIKE '%".$http_vars[SEARCH_STRING]."%')";
		$whereClause .= " OR (_sub_category_1 LIKE '%".$http_vars[SEARCH_STRING]."%')";
		$whereClause .= " OR (_sub_category_2 LIKE '%".$http_vars[SEARCH_STRING]."%')";
		$whereClause .= " OR (_sub_category_3 LIKE '%".$http_vars[SEARCH_STRING]."%')";
		
		$sql = "SELECT _key FROM $c_CategoryTable WHERE _name LIKE '%".$http_vars[SEARCH_STRING]."%'";

		$result = sql_query ( $sql, $dbi );
		
		if ( $result )
			{
			while ( $row = sql_fetch_array ( $result, $dbi ) )
				{
				$whereClause .= " OR (_category='$row[_key]')";
				$whereClause .= " OR (_sub_category_1='$row[_key]')";
				$whereClause .= " OR (_sub_category_2='$row[_key]')";
				$whereClause .= " OR (_sub_category_3='$row[_key]')";
				}
			}
		
		if ( $php_nuke_environment )
			{
			$sql = "SELECT uname FROM $user_prefix"."_users WHERE (name LIKE '%".$http_vars[SEARCH_STRING]."%') ";
	
			$result = sql_query ( $sql, $dbi );
			
			if ( $result )
				{
				while ( list ( $uname ) = sql_fetch_array ( $result, $dbi ) )
					{
					$whereClause .= " OR (_user='$uname') OR (_user LIKE '%".$http_vars[SEARCH_STRING]."%')";
					}
				}
			}
		
		$whereClause .= " OR (_description LIKE '%".$http_vars[SEARCH_STRING]."%'))";
		$http_vars['FIND_CAT'] = "";
		}
	elseif ( $http_vars['FIND_CAT'] )
		{
		if ( preg_match ( "/(.*)\%$/", $http_vars['FIND_CAT'] ) )
			{
			$whereClause = "WHERE ((_category LIKE '".$http_vars['FIND_CAT']."')\n";
			$whereClause .= " OR (_sub_category_1 LIKE '".$http_vars['FIND_CAT']."')\n";
			$whereClause .= " OR (_sub_category_2 LIKE '".$http_vars['FIND_CAT']."')\n";
			$whereClause .= " OR (_sub_category_3 LIKE '".$http_vars['FIND_CAT']."'))\n";
			}
		else
			{
			if ( $http_vars['FIND_CAT'] != "All_Categories" )
				{
				$whereClause = "WHERE ((_category='".$http_vars['FIND_CAT']."')\n";
				$whereClause .= " OR (_sub_category_1='".$http_vars['FIND_CAT']."')\n";
				$whereClause .= " OR (_sub_category_2='".$http_vars['FIND_CAT']."')\n";
				$whereClause .= " OR (_sub_category_3='".$http_vars['FIND_CAT']."'))\n";
				}
			}
		}
	
	if ( !$editor )
		{
		$whereClause2 = "(_active='1')";
		if ( $whereClause )
			{
			$whereClause .= " AND $whereClause2";
			}
		else
			{
			$whereClause = "WHERE $whereClause2";
			}
		}
	
	if ( $http_vars[FIND_CAT] == "Inactive_Links" )
		{
		$whereClause = "WHERE _active=0";
		}
	
	$dbi = sql_connect( $dbhost, $dbuname, $dbpass, $dbname );
	
	$sql = "SELECT * FROM $c_LinksTable $whereClause ORDER BY $first_Sort, $second_Sort, $third_Sort";
	
	$result = sql_query ( $sql, $dbi );
	
//	echo "DisplayLinks(): \$dbhost = \"$dbhost\", \$dbname = \"$dbname\", \$sql = \"$sql\", \$dbi = \"$dbi\", \$result = \"$result\", sql_error = \"".sql_error()."\"<br />\n";
	
	if ( $result )
		{
		$numRows = sql_num_rows ( $result, $dbi );
		print ( "<table cellpadding=\"0\" cellspacing=\"0\" style=\"margin:0;padding:0;border:none;width:100%\">\n" );
			print ( "<tr>\n" );
				print ( "<td  colspan=\"3\" style=\"text-align: center\"><hr /></td>\n" );
			print ( "</tr>\n" );
			if ( $http_vars[SHOW_OPTIONS] )
				{
				if ( $editor )
					{
					if ( can_admin ( "EditCategories" ) )
						{
						print ( "<tr>\n" );
							print ( "<td style=\"text-align:left\">" );
								print ( "<form action=\"$target_File\" method=\"$editormethod\"><div>\n" );
									print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
									print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
									print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
									print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
									print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
									print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
									print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
									print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
									print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
									print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"EditCategories\" />\n" );
									print ( "<input type=\"submit\" value=\""._EDIT_CATEGORIES."\" />\n" );
								print ( "</div></form>" );
							print ( "</td>\n" );
							print ( "<td style=\"text-align:left\">&nbsp;</td>\n" );
							print ( "<td  style=\"text-align: right\">" );
								if ( $numRows )
									{
									print ( "<form action=\"$target_File\" method=\"$editormethod\" onSubmit=\"return confirm('"._TEST_ALL_CONFIRM."')\"><div>\n" );
										print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
										print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
										print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
										print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
										print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
										print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
										print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"TestAllFoundLinks\" />\n" );
										print ( "<input type=\"submit\" value=\""._TEST_ALL."\" />\n" );
									print ( "</div></form>" );
									}
								else
									{
									print ("&nbsp;" );
									}
							print ( "</td>\n" );
						print ( "</tr>\n" );
					
						print ( "<tr>\n" );
							print ( "<td  colspan=\"3\" style=\"text-align: center\"><hr /></td>\n" );
						print ( "</tr>\n" );
						}
						
					if ( $http_vars[TEST_RESULTS] )
						{
						print ( "<tr>\n" );
							print ( "<td  colspan=\"3\" style=\"text-align: center\"><a href=\"#Test_Results\">"._TEST_RESULTS."</a></td>\n" );
						print ( "</tr>\n" );
						print ( "<tr>\n" );
							print ( "<td  colspan=\"3\" style=\"text-align: center\"><hr /></td>\n" );
						print ( "</tr>\n" );
						}
					
					print ( "<tr>\n" );
						print ( "<td  colspan=\"3\" style=\"text-align: center\">" );
							print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none;width:100%\">\n" );
								print ( "<tr>\n" );
									print ( "<td  style=\"text-align: center\">" );
										print ( "<form action=\"$target_File\" method=\"$editormethod\" onSubmit=\"return confirm('"._DELETE_ALL_CONFIRM."')\"><div>\n" );
											print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
											print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
											print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
											print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
											print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
											print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
											print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
											print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
											print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
											print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"DeleteFoundLinks\" />\n" );
											print ( "<input type=\"submit\" value=\""._DELETE_ALL."\" />\n" );
										print ( "</div></form>" );
									print ( "</td>\n" );
								print ( "</tr>\n" );
							print ( "</table>\n" );
						print ( "</td>\n" );
					print ( "</tr>\n" );
					print ( "<tr>\n" );
						print ( "<td  colspan=\"3\" style=\"text-align: center\"><hr /></td>\n" );
					print ( "</tr>\n" );
					}
				
				print ( "<tr>\n" );
					print ( "<td  colspan=\"3\">\n" );
						print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none;width:100%\">\n" );
							print ( "<tr>\n" );
								print ( "<td  valign=\"top\">\n" );
									print ( "<form id=\"SortBy\" action=\"$target_File\"" );
										if ( $editor )
											{
											print ( " method=\"$editormethod\"><div>\n" );
											}
										else
											{
											print ( " method=\"$usermethod\"><div>\n" );
											}

										print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
										print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
										print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
										if ( $http_vars[ACTION] )
											{
											print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
											}
										if ( $http_vars[DISP_COUNT] )
											{
											print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
											}
										if ( $http_vars[SEARCH_STRING] )
											{
											print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
											}
										if ( $http_vars[FIND_CAT] )
											{
											print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
											}
										if ( $http_vars[SHOW_OPTIONS] )
											{
											print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
											}
										print ( "<label for=\"SORT_BY\"><strong>"._SORT_BY."</strong></label> <select title=\""._SORT_BY_TITLE."\" id=\"SORT_BY\" name=\"SORT_BY\" onchange=\"document.getElementById('SortBy').submit()\">\n" );
											print ( "<option value=\"Timestamp\"" );
											if ( ($http_vars[SORT_BY] == "Timestamp")
												|| ($http_vars[FIND_CAT] && ($http_vars[SORT_BY] == "Category_Time")) )
												{
												print ( " selected=\"selected\"" );
												}
											print ( ">"._SORT_OPTION_TIMESTAMP."</option>\n" );
												
											print ( "<option value=\"Title\"" );
											if ( ($http_vars[SORT_BY] == "Title")
												|| ($http_vars[FIND_CAT] && ($http_vars[SORT_BY] == "Category_Title")) )
												{
												print ( " selected=\"selected\"" );
												}
											print ( ">"._SORT_OPTION_TITLE."</option>\n" );
											
											if ( $php_nuke_environment )
												{
												print ( "<option value=\"Submitter\"" );
												if ( $http_vars[SORT_BY] == "Submitter" )
													{
													print ( " selected=\"selected\"" );
													}
												print ( ">"._SORT_OPTION_SUBMITTER."</option>\n" );
												}
											
											if ( !$http_vars[FIND_CAT] || preg_match ( "/(.*)\%$/", $http_vars[FIND_CAT] ) )
												{
												print ( "<option value=\"Category_Time\"" );
												if ( $http_vars[SORT_BY] == "Category_Time" )
													{
													print ( " selected=\"selected\"" );
													}
												print ( ">"._SORT_OPTION_CAT_TIME."</option>\n" );
												
												print ( "<option value=\"Category_Title\"" );
												if ( $http_vars[SORT_BY] == "Category_Title" )
													{
													print ( " selected=\"selected\"" );
													}
												print ( ">"._SORT_OPTION_CAT_TITLE."</option>\n" );
												}
											else
												{
												if ( $http_vars[SORT_BY] == "Category_Title" )
													{
													$first_Sort = "_title";
													$second_Sort = "_category";
													$third_Sort = "_timestamp DESC";
													}
												
												if ( $http_vars[SORT_BY] == "Category_Time" )
													{
													$first_Sort = "_timestamp DESC";
													$second_Sort = "_category";
													$third_Sort = "_title";
													}
												}
											
										print ( "</select>\n" );
									print ( "</div></form>" );
								print ( "</td>\n" );
								print ( "<td  valign=\"top\" style=\"text-align: right\">" );
									print ( "<form id=\"ChangeNumResults\" action=\"$target_File\"" );
									if ( $editor )
										{
										print ( " method=\"$editormethod\"><div>\n" );
										}
									else
										{
										print ( " method=\"$usermethod\"><div>\n" );
										}

									print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
									print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
									print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
									if ( $http_vars[ACTION] )
										{
										print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
										}
									if ( $http_vars[SORT_BY] )
										{
										print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
										}
									if ( $http_vars[SEARCH_STRING] )
										{
										print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
										}
									if ( $http_vars[FIND_CAT] )
										{
										print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
										}
									if ( $http_vars[SHOW_OPTIONS] )
										{
										print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
										}
									print ( "<label for=\"DISP_COUNT\"><strong>"._RESULTS_PER_PAGE."</strong></label> <select title=\""._RESULTS_PER_PAGE_TITLE."\" id=\"DISP_COUNT\" name=\"DISP_COUNT\" onchange=\"document.getElementById('ChangeNumResults').submit()\">\n" );
									
									$oval = 5;
									
									if ( ($http_vars[DISP_COUNT] != -1) && ($http_vars[DISP_COUNT] < $oval) )
										{
										$http_vars[DISP_COUNT] = $oval;
										}
									
									while ( $oval <= 100 )
										{
										print ( "<option value=\"$oval\"" );
										$inc = 10;
										if ( $oval < 10 )
											{
											$inc = 10 - $oval;
											}
										if ( ($http_vars[DISP_COUNT] >= $oval) && ($http_vars[DISP_COUNT] < ($oval + $inc)) )
											{
											print ( " selected=\"selected\"" );
											}
										print ( ">$oval</option>\n" );
										$oval += $inc;
										}
									print ( "<option value=\"-1\"" );
									if ( $http_vars[DISP_COUNT] == -1 ) {
										print ( " selected=\"selected\"" );
										}
									print ( ">"._RESULTS_PER_PAGE_ALL."</option>\n" );
									print ( "</select>\n" );
									print ( "</div></form>" );
								print ( "</td>\n" );
							print ( "</tr>\n" );
							print ( "<tr>\n" );
								print ( "<td  colspan=\"3\" style=\"text-align: center\">\n" );
									print ( "<table cellpadding=\"0\" cellspacing=\"0\" style=\"border:none;width:100%\">\n" );
										print ( "<tr>\n" );
											print ( "<td  colspan=\"2\" style=\"text-align: center\"><hr /></td>\n" );
										print ( "</tr>\n" );
										print ( "<tr>\n" );
												print ( "<td  style=\"text-align: center\" colspan=\"2\">" );
													print ( "<form id=\"SearchLinks\" action=\"$target_File\"" );
														if ( $editor )
															{
															print ( " method=\"$editormethod\"><div>\n" );
															}
														else
															{
															print ( " method=\"$usermethod\"><div>\n" );
															}

														print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
														print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"1\" />\n" );
														print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
														print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
														if ( $http_vars[SORT_BY] )
															{
															print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
															}
														if ( $http_vars[DISP_COUNT] )
															{
															print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
															}
														print ( "<script type=\"text/javascript\">\n<!-- // Hide from old browsers\n" );
															print ( "function ResetCategory(){\n" );
																print ( "document.getElementById('SelectCategory').FIND_CAT.selectedIndex=0;return true;\n" );
															print ( "}\n" );
														print ( "// Done Hiding -->\n</script>\n" );
														print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none\">\n" );
															print ( "<tr>\n" );
																print ( "<td>\n" );
																	print ( "<label for=\"SEARCH_STRING\"><strong>"._SEARCH_LINKS."</strong></label>" );
																print ( "</td>\n" );
																print ( "<td>\n" );
																	print ( "<input maxlength=\"128\" size=\"50\" type=\"text\" id=\"SEARCH_STRING\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" title=\""._SEARCH_TITLE."\" onchange=\"ResetCategory()\" onkeypress=\"ResetCategory()\" />" );
																print ( "</td>\n" );
																print ( "<td>\n" );
																	print ( "<input type=\"submit\" value=\""._GO."\" />\n" );
																print ( "</td>\n" );
															print ( "</tr>\n" );
													print ( "</table>" );
												print ( "</div></form>" );
											print ( "</td>\n" );
										print ( "</tr>\n" );
										print ( "<tr>\n" );
											print ( "<td  colspan=\"2\" style=\"text-align: center\"><hr /></td>\n" );
										print ( "</tr>\n" );
										print ( "<tr>\n" );
											print ( "<td  style=\"text-align: center\"><form id=\"SelectCategory\" action=\"$target_File\"" );
												if ( $editor )
													{
													print ( " method=\"$editormethod\"><div>\n" );
													}
												else
													{
													print ( " method=\"$usermethod\"><div>\n" );
													}
												print ( "<label for=\"FIND_CAT\"><strong>"._CATEGORY."</strong></label>&nbsp;" );
												print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
												print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
												print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
												if ( $http_vars[ACTION] )
													{
													print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
													}
												if ( $http_vars[DISP_COUNT] )
													{
													print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
													}
												if ( $http_vars[SHOW_OPTIONS] )
													{
													print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
													}
												if ( $http_vars[SORT_BY] )
													{
													print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
													}
												PrintCategoryPopup ( $http_vars[FIND_CAT], $editor, 0, $http_vars, "FIND_CAT", "document.getElementById('SearchLinks').SEARCH_STRING.value=''" );
											print ( "</div></form></td>\n" );
										print ( "</tr>\n" );
									print ( "</table>" );
								print ( "</td>\n" );
							print ( "</tr>\n" );
						print ( "</table>\n" );
					print ( "</td>\n" );
				print ( "</tr>\n" );
				print ( "<tr>\n" );
					print ( "<td  colspan=\"3\" style=\"text-align: center\"><hr /></td>\n" );
				print ( "</tr>\n" );
				print ( "<tr>\n" );
					print ( "<td  colspan=\"3\" style=\"text-align: center\">" );
						print ( "<form action=\"$target_File\"" );
							if ( $editor )
								{
								print ( " method=\"$editormethod\"><div>\n" );
								}
							else
								{
								print ( " method=\"$usermethod\"><div>\n" );
								}
							print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
							print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
							print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
							if ( $http_vars[ACTION] )
								{
								print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
								}
							if ( $http_vars[DISP_COUNT] )
								{
								print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
								}
							if ( $http_vars[FIND_CAT] )
								{
								print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
								}
							if ( $http_vars[SORT_BY] )
								{
								print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
								}
							if ( $http_vars[START_COUNT] )
								{
								print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
								}
							print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"\" />\n" );
							print ( "<input title=\""._FEWER_OPTIONS_TITLE."\" type=\"submit\" value=\""._FEWER_OPTIONS."\" />\n" );
						print ( "</div></form>\n" );
					print ( "</td>\n" );
				print ( "</tr>\n" );
				}
			else
				{
				print ( "<tr>\n" );
					print ( "<td  colspan=\"3\" style=\"text-align: center\">" );
						print ( "<table style=\"border:none;width:100%\" cellpadding=\"3\" cellspacing=\"0\">\n" );
							print ( "<tr>\n" );
								print ( "<td style=\"text-align:left\">" );
									print ( "<form id=\"SelectCategory\" action=\"$target_File\"" );
										if ( $editor )
											{
											print ( " method=\"$editormethod\"><div>\n" );
											}
										else
											{
											print ( " method=\"$usermethod\"><div>\n" );
											}

										print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
										print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
										print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
										if ( $http_vars[ACTION] )
											{
											print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
											}
										if ( $http_vars[DISP_COUNT] )
											{
											print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
											}
										if ( $http_vars[SHOW_OPTIONS] )
											{
											print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
											}
										if ( $http_vars[SORT_BY] )
											{
											print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
											}
										print ( "<label for=\"FIND_CAT\"><strong>"._CATEGORY."</strong></label>&nbsp;" );
										PrintCategoryPopup ( $http_vars[FIND_CAT], $editor, 0, $http_vars );
									print ( "</div></form>\n" );
								print ( "</td>\n" );
								print ( "<td  style=\"text-align: right\">\n" );
									print ( "<form action=\"$target_File\"" );
										if ( $editor )
											{
											print ( " method=\"$editormethod\"><div>\n" );
											}
										else
											{
											print ( " method=\"$usermethod\"><div>\n" );
											}

										print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
										print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
										print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
										if ( $http_vars[ACTION] )
											{
											print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
											}
										if ( $http_vars[DISP_COUNT] )
											{
											print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
											}
										if ( $http_vars[FIND_CAT] )
											{
											print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
											}
										if ( $http_vars[SORT_BY] )
											{
											print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
											}
										if ( $http_vars[START_COUNT] )
											{
											print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
											}
										print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"1\" />\n" );
										print ( "<input title=\""._MORE_OPTIONS_TITLE."\" type=\"submit\" value=\""._MORE_OPTIONS."\" />\n" );
									print ( "</div></form>\n" );
								print ( "</td>\n" );
							print ( "</tr>\n" );
						print ( "</table>\n" );
					print ( "</td>\n" );
				print ( "</tr>\n" );
				}
		
		print ( "<tr>\n" );
			print ( "<td  colspan=\"3\" style=\"text-align: center\"><hr /></td>\n" );
		print ( "</tr>\n" );
		print ( "<tr>\n" );
			print ( "<td  colspan=\"3\" style=\"text-align: center\"><h3>($numRows "._LINKS_FOUND.")</h3></td>\n" );
		print ( "</tr>\n" );
		print ( "<tr>\n" );
			print ( "<td  colspan=\"3\" style=\"text-align: center\"><hr /></td>\n" );
		print ( "</tr>\n" );
		
		if ( $numRows > 0 )
			{
			$ended = "";
			$rowCount = 0;
			$currentCat = "";
			
			print ( "<tr>\n" );
				print ( "<td  colspan=\"3\">\n" );
					print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none\">\n" );
					while ( $row = sql_fetch_array ( $result, $dbi ) )
						{
						$rowCount++;
						if ( ($rowCount > $startCount)
							&& (($rowCount <= $endCount)  || !$endCount)
							&& (($foundCat && $searchCat) || !$searchCat) )
							{
							$lastRow = $rowCount;
							$userinfo = getusrinfo ( $user );
							$usereditor = "";
							if ( $php_nuke_environment && !$moderate_links && ($userinfo[uname] == $row[_user]) )
								{
								$usereditor = 1;
								}
							print ( "<tr>\n" );
								print ( "<td  colspan=\"2\">" );
								print ( "<form id=\"Link_$rowCount\" action=\"$target_File\"" );
									if ( $editor )
										{
										print ( " method=\"$editormethod\"><div>\n" );
										}
									else
										{
										print ( " method=\"$usermethod\"><div>\n" );
										}
									print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
									print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
									print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
									if ( $http_vars[ACTION] )
										{
										print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
										}
									if ( $http_vars[SORT_BY] )
										{
										print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
										}
									if ( $http_vars[SEARCH_STRING] )
										{
										print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
										}
									if ( $http_vars[SHOW_OPTIONS] )
										{
										print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
										}

									$uname = $row[_user];
									$title = $row[_title];
									$url = $row[_url];
									$description = "";
									if ( $row[_description] )
										{
										$description = "<p class=\"first\">".ereg_replace ( "\n", "</p>\n<p>", $row[_description] )."</p>";
										}
									
									if ( $php_nuke_environment && $uname )
										{
										$sql_u = "SELECT name FROM $user_prefix"."_users where uname='$uname'";
										$uresult = sql_query ( $sql_u, $dbi );
										if ( !$uresult )
											{
											$sql_u = "SELECT name FROM $user_prefix"."_users where username='$uname'";
											$uresult = sql_query ( $sql_u, $dbi );
											}
										
										if ( $uresult )
											{
											list ( $name ) = sql_fetch_array ( $uresult, $dbi);
											
											if ( !$name )
												{
												$name = $uname;
												}
											
											if ( $name )
												{
												$description .= "<p>"._SUBMITTED_BY." <a href=\"modules.php?name=Your_Account&amp;op=userinfo&amp;uname=$uname\">$name</a></p>\n";
												}
											}
										}
									
									// Make sure there's a proper URL here.
									$url = htmlentities ( $url );

									if ( ($row[_category] != "local") && !preg_match ( "|^.*:/.*|i", $url ) && $replace_URL_Header )
										{
										$url = "http://".$url;
										}
									
									if ( !$row[_active] )
										{
										print ( "<em>" );
										}
									else
										{
										print ( "<strong>" );
										}
									print ( "<a  class=\"link_out\" title=\"$url\" href=\"$url\">".htmlentities ( $title )."</a>" );
									if ( !$row[_active] )
										{
										print ( "</em>" );
										}
									else
										{
										print ( "</strong>" );
										}
									$catfind = $http_vars[FIND_CAT];
									if ( !$http_vars[FIND_CAT]	// We display the category if we aren't looking at one specific category
										|| ($http_vars[FIND_CAT] == "All_Categories")
										|| ($http_vars[FIND_CAT] == "Inactive_Links")
										|| preg_match ( "/(.*)\%$/", $http_vars[FIND_CAT] ) )
										{
										$sql2 = "SELECT * FROM $c_CategoryTable WHERE _key='$row[_category]'";
										$result2 = sql_query ( $sql2, $dbi );
										if ( $result2 )
											{
											$row2 = sql_fetch_array ( $result2, $dbi );
											if ( !$row2[_name] )
												{
												$sql2 = "SELECT * FROM $c_CategoryTable WHERE _key='Misc'";
												$result2 = sql_query ( $sql2, $dbi );
												if ( $result2 )
													{
													$row2 = sql_fetch_array ( $result2, $dbi );
													if ( !$row2[_name] )
														{
														$row2[_name] = ""._NO_CATEGORY."";
														}
													}

												if ( !$row2[_description] )
													{
													$row2[_description] = ""._NO_CATEGORY_DESC."";
													}
												}
											if ( $editor || ($row[_category] != "local") )
												{
												$row2[_description] .= ""._CLICK_OPENS_NEW_WINDOW."";
												}
											$catfind = $row2[_key];
											print ( " (<a title=\"".htmlentities ( $row2[_description] )."\" href=\"javascript:if (confirm('".addslashes($row2[_description])."\\n"._CATEGORY_QUERY."'))document.getElementById('Link_$rowCount').submit()\">$row2[_name]</a>)\n" );
											}
										}
									print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$catfind\" />\n" );
									print ( "</div></form>\n" );
								print ( "</td>\n" );
								if ( $editor || $usereditor )
									{
									print ( "<script type=\"text/javascript\">\n<!-- // Hide from old browsers\n" );
										print ( "function ConfirmDelete(nam){\n" );
											print ( "return confirm('"._DEL_CONF."'+nam+'"._QUESTION_MARK."');\n" );
										print ( "}\n" );
									print ( "// Done Hiding -->\n</script>\n" );
									
									print ( "<td " );
									if ( $description )
										{
										print ( " rowspan=\"2\"" );
										}
									print ( ">" );
									print ( "<form action=\"$target_File\" onSubmit=\"return ConfirmDelete('$title')\"" );
										if ( $editor )
											{
											print ( " method=\"$editormethod\"><div>\n" );
											}
										else
											{
											print ( " method=\"$usermethod\"><div>\n" );
											}
										print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
										print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
										print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
										print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
										print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
										print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
										print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
										print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"LINKURL\" value=\"$url\" />\n" );
										print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"Delete\" />\n" );
										print ( "<input type=\"submit\" value=\""._DELETE."\" />\n" );
									print ( "</div></form>" );
									print ( "</td>\n" );

									print ( "<td " );
									if ( $description )
										{
										print ( " rowspan=\"2\"" );
										}
									print ( ">" );
									print ( "<form action=\"$target_File\"" );
										if ( $editor )
											{
											print ( " method=\"$editormethod\"><div>\n" );
											}
										else
											{
											print ( " method=\"$usermethod\"><div>\n" );
											}
										print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
										print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
										print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
										print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
										print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
										print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
										print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
										print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
										print ( "<input type=\"hidden\" name=\"LINKURL\" value=\"$url\" />\n" );
										print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"Edit\" />\n" );
										print ( "<input type=\"submit\" value=\""._EDIT."\" />\n" );
									print ( "</div></form>" );
									print ( "</td>\n" );

									print ( "<td " );
									if ( $description )
										{
										print ( " rowspan=\"2\"" );
										}
									print ( ">" );
									if ( $editor )
										{
										print ( "<form action=\"$target_File\" method=\"$editormethod\"><div>" );
											print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
											print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
											print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
											print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
											print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
											print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
											print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
											print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
											print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
											print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
											print ( "<input type=\"hidden\" name=\"LINKURL\" value=\"$url\" />\n" );
											print ( "<input type=\"hidden\" name=\"LINK_ACTION\"" );
											if ( !$row[_active] )
												{
												print ( " value=\"Activate\">\n<input type=\"submit\" value=\""._ACTIVATE."\" />\n" );
												}
											else
												{
												print ( " value=\"Deactivate\">\n<input type=\"submit\" value=\""._DEACTIVATE."\" />\n" );
												}
										print ( "</div></form>" );
										}
									else
										{
										print ( "&nbsp;" );
										}
									print ( "</td>\n" );
									}
								else
									{
									print ( "<td " );
									if ( $description )
										{
										print ( " rowspan=\"2\"" );
										}
									print ( ">&nbsp;</td>\n" );
									print ( "<td " );
									if ( $description )
										{
										print ( " rowspan=\"2\"" );
										}
									print ( ">&nbsp;</td>\n" );
									print ( "<td " );
									if ( $description )
										{
										print ( " rowspan=\"2\"" );
										}
									print ( ">&nbsp;</td>\n" );
									}
							print ( "</tr>\n" );
							if ( $description )
								{
								print ( "<tr>\n" );
									print ( "<td  style=\"width:50px\">&nbsp;</td>\n" );
									print ( "<td  style=\"width:500px\">$description</td>\n" );
								print ( "</tr>\n" );
								}
							print ( "<tr>\n" );
								print ( "<td  colspan=\"5\" style=\"text-align: center\"><hr style=\"width:80%\" /></td>\n" );
							print ( "</tr>\n" );
							}
						}
					
					if ( $lastRow == $rowCount )
						{
						$ended = 1;
						}
					
					print ( "</table>\n" );
					}
				else
					{
					print ( "<td  colspan=\"3\" style=\"text-align: center\">\n" );
					print ( "<h2>"._NO_LINKS."</h2>" );
					}
				print ( "</td>\n" );
			print ( "</tr>\n" );
			print ( "<tr>\n" );
				print ( "<td  colspan=\"3\" style=\"text-align: center\"><hr /></td>" );
			print ( "</tr>\n" );
			print ( "<tr>\n" );
				print ( "<td  valign=\"top\" style=\"width:125px\">\n" );
				if ( $startCount && ($rowCount > 5))
					{
					print ( "<form id=\"BackLinks\" action=\"$target_File\"" );
					if ( $editor )
						{
						print ( " method=\"$editormethod\"><div>\n" );
						}
					else
						{
						print ( " method=\"$usermethod\"><div>\n" );
						}

					$stCount = $startCount - $http_vars[DISP_COUNT];
					if ( $stCount < 0 )
						{
						$stCount = 0;
						}
					print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
					print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$stCount\" />\n" );
					print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
					print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
					if ( $http_vars[ACTION] )
						{
						print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
						}
					if ( $http_vars[DISP_COUNT] )
						{
						print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
						}
					if ( $http_vars[SEARCH_STRING] )
						{
						print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
						}
					if ( $http_vars[FIND_CAT] )
						{
						print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
						}
					if ( $http_vars[SHOW_OPTIONS] )
						{
						print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
						}
					if ( $http_vars[SORT_BY] )
						{
						print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
						}
					print ( "<a href=\"javascript:document.getElementById('BackLinks').submit()\">"._PREV."</a>" );
					print ( "</div></form>" );
					}
				else
					{
					print ( "&nbsp;" );
					}
				print ( "</td>\n" );
				print ( "<td  valign=\"top\" style=\"text-align: center;width:100%\">\n" );
					if ( $rowCount && $http_vars[DISP_COUNT] && $endCount )
						{
						$totalPages = intval (($rowCount + ($http_vars[DISP_COUNT] - 1)) / $http_vars[DISP_COUNT]);
						
						$pages = $totalPages;
						
						if ( $pages > 1 )
							{
							$initPage = 1;
							
							$page_threshold = 11;
							
							if ( $pages > $page_threshold )
								{
								$initPage =  intval ($startCount / $http_vars[DISP_COUNT]) - (intval ($page_threshold / 2) - 1);
								
								if ( $initPage > ($pages - $page_threshold) )
									{
									$initPage = (($pages - $page_threshold) + 1);
									}
								
								if ( $initPage < 1 )
									{
									$initPage = 1;
									}
								
								$pagesTemp = $initPage + ($page_threshold - 1);
								
								if ( $pagesTemp < $pages )
									{
									$pages = $pagesTemp;
									}
								}
							
							print ( "<table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none;margin-left:auto;margin-right:auto\">\n<tr valign=\"top\">\n<td>Page:</td>" );
								if ( ($initPage != 1) && ($totalPages > $page_threshold) )
									{
									print ( "<td style=\"text-align:left\">" );
										print ( "<form id=\"first_Page\" action=\"$target_File\"" );
										if ( $editor )
											{
											print ( " method=\"$editormethod\"><div>\n" );
											}
										else
											{
											print ( " method=\"$usermethod\"><div>\n" );
											}
	
										print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
										print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"1\" />\n" );
										print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
										print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
										if ( $http_vars[ACTION] )
											{
											print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
											}
										if ( $http_vars[DISP_COUNT] )
											{
											print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
											}
										if ( $http_vars[SEARCH_STRING] )
											{
											print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
											}
										if ( $http_vars[FIND_CAT] )
											{
											print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
											}
										if ( $http_vars[SHOW_OPTIONS] )
											{
											print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
											}
										if ( $http_vars[SORT_BY] )
											{
											print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
											}
									
									if ( $startCount > $http_vars[DISP_COUNT] )
										{
										print ( "<strong><a href=\"javascript:document.getElementById('first_Page').submit()\">1</a>&nbsp;...</strong>" );
										}
									else
										{
										print ( "<strong>1&nbsp;...</strong>" );
										}
									print ( "</div></form>" );
								print ( "</td>\n" );
								}
							for ( $cnt = $initPage; $cnt <= $pages; $cnt++ )
								{
								print ( "<td  style=\"text-align: center\">" );
								$upper = ($http_vars[DISP_COUNT] * $cnt);
								$lower = $upper - $http_vars[DISP_COUNT];
								if ( $upper > $rowCount )
									{
									$upper = $rowCount;
									}
								if ( ($startCount < $lower) || ($startCount >= $upper) )
									{
									$st = ($http_vars[DISP_COUNT] * $cnt - 1);
									print ( "<form id=\"Page_$cnt\" action=\"$target_File\"" );
									if ( $editor )
										{
										print ( " method=\"$editormethod\"><div>\n" );
										}
									else
										{
										print ( " method=\"$usermethod\"><div>\n" );
										}

									print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
									print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$st\" />\n" );
									print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
									print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
									if ( $http_vars[ACTION] )
										{
										print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
										}
									if ( $http_vars[DISP_COUNT] )
										{
										print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
										}
									if ( $http_vars[SEARCH_STRING] )
										{
										print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
										}
									if ( $http_vars[FIND_CAT] )
										{
										print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
										}
									if ( $http_vars[SHOW_OPTIONS] )
										{
										print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
										}
									if ( $http_vars[SORT_BY] )
										{
										print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
										}
									print ( "<a href=\"javascript:document.getElementById('Page_$cnt').submit()\"><strong>$cnt</strong></a>" );
									print ( "</div></form>" );
									}
								else
									{
									print ( "<strong>$cnt</strong>" );
									}
								print ( "</td>\n" );
								}
								if ( !$ended && ($totalPages > $page_threshold) )
									{
									print ( "<td style=\"text-align:left\">" );
										print ( "<form id=\"Last_Page\" action=\"$target_File\"" );
										if ( $editor )
											{
											print ( " method=\"$editormethod\"><div>\n" );
											}
										else
											{
											print ( " method=\"$usermethod\"><div>\n" );
											}
	
										$end_target = (($totalPages - 1) * $http_vars[DISP_COUNT]) + 1;
										print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
										print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$end_target\" />\n" );
										print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
										print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
										if ( $http_vars[ACTION] )
											{
											print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
											}
										if ( $http_vars[DISP_COUNT] )
											{
											print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
											}
										if ( $http_vars[SEARCH_STRING] )
											{
											print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
											}
										if ( $http_vars[FIND_CAT] )
											{
											print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
											}
										if ( $http_vars[SHOW_OPTIONS] )
											{
											print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
											}
										if ( $http_vars[SORT_BY] )
											{
											print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
											}
									if ( !$ended )
										{
										print ( "<strong>...&nbsp;<a href=\"javascript:document.getElementById('Last_Page').submit()\">$totalPages</a></strong>" );
										}
									else
										{
										print ( "<strong>...&nbsp;$totalPages</strong>" );
										}
									print ( "</div></form>" );
								print ( "</td>\n" );
								}
							print ( "</tr>\n</table>\n" );
							}
						else
							{
							print ( "&nbsp;" );
							}
						}
					else
						{
						print ( "&nbsp;" );
						}
				print ( "</td>\n" );
				print ( "<td  valign=\"top\" style=\"text-align: right;width:125px\">\n" );
				if ( !$ended && ($rowCount > 5))
					{
					print ( "<form id=\"NextLinks\" action=\"$target_File\"" );
					if ( $editor )
						{
						print ( " method=\"$editormethod\"><div>\n" );
						}
					else
						{
						print ( " method=\"$usermethod\"><div>\n" );
						}

					$stCount = $startCount + $http_vars[DISP_COUNT];
					if ( $stCount < 0 )
						{
						$stCount = 0;
						}
					print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
					print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$stCount\" />\n" );
					print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
					print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
					if ( $http_vars[ACTION] )
						{
						print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
						}
					if ( $http_vars[DISP_COUNT] )
						{
						print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
						}
					if ( $http_vars[SEARCH_STRING] )
						{
						print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
						}
					if ( $http_vars[FIND_CAT] )
						{
						print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
						}
					if ( $http_vars[SHOW_OPTIONS] )
						{
						print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
						}
					if ( $http_vars[SORT_BY] )
						{
						print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
						}
					print ( "<a href=\"javascript:document.getElementById('NextLinks').submit()\">"._NEXT."</a>" );
					print ( "</div></form>" );
					}
				else
					{
					print ( "&nbsp;" );
					}
				print ( "</td>\n" );
			print ( "</tr>\n" );
		print ( "</table>\n" );
		if ( can_submit ( ) )
			{
			print ( "<div style=\"text-align: center\">\n<table style=\"border:none\" cellpadding=\"5\" cellspacing=\"0\">\n" );
				print ( "<tr>\n" );
					print ( "<td style=\"text-align: center\"><form action=\"$target_File\"" );
						if ( $editor )
							{
							print ( " method=\"$editormethod\"><div>\n" );
							}
						else
							{
							print ( " method=\"$usermethod\"><div>\n" );
							}
							print ( "<input type=\"hidden\" name=\"NAME\" value=\"$mod_Name\" />\n" );
							print ( "<input type=\"hidden\" name=\"START_COUNT\" value=\"$http_vars[START_COUNT]\" />\n" );
							print ( "<input type=\"hidden\" name=\"DISP_COUNT\" value=\"$http_vars[DISP_COUNT]\" />\n" );
							print ( "<input type=\"hidden\" name=\"LINK_ACTION\" value=\"SubmitNewLink\" />\n" );
							print ( "<input type=\"hidden\" name=\"SORT_BY\" value=\"$http_vars[SORT_BY]\" />\n" );
							print ( "<input type=\"hidden\" name=\"ACTION\" value=\"$http_vars[ACTION]\" />\n" );
							print ( "<input type=\"hidden\" name=\"LOGIN\" value=\"$http_vars[LOGIN]\" />\n" );
							print ( "<input type=\"hidden\" name=\"CPASS\" value=\"$http_vars[CPASS]\" />\n" );
	
							if ( $http_vars[SEARCH_STRING] )
								{
								print ( "<input type=\"hidden\" name=\"SEARCH_STRING\" value=\"$http_vars[SEARCH_STRING]\" />\n" );
								}
							if ( $http_vars[FIND_CAT] )
								{
								print ( "<input type=\"hidden\" name=\"FIND_CAT\" value=\"$http_vars[FIND_CAT]\" />\n" );
								}
							if ( $http_vars[SHOW_OPTIONS] )
								{
								print ( "<input type=\"hidden\" name=\"SHOW_OPTIONS\" value=\"$http_vars[SHOW_OPTIONS]\" />\n" );
								}
							print ( "<input type=\"submit\" value=\""._SUBMIT_A_NEW_LINK."\" />\n" );
						print ( "</div></form></td>\n" );
				print ( "</tr>\n" );
			print ( "</table>\n</CENTER\n" );
			}
		if ( $http_vars[TEST_RESULTS] )
			{
			print ( "<div style=\"text-align: center\"><a name=\"Test_Results\"></a><table cellpadding=\"3\" cellspacing=\"0\" style=\"border:none\">\n" );
				print ( $http_vars[TEST_RESULTS] );
			print ( "</table>\n</td>\n</div>\n" );
			}
		}
	}

/***************************************************************************
	Start of main context code
*/
Links_HeaderDisplay ( );

if ( !$http_vars[DISP_COUNT] )
	{
	$http_vars[DISP_COUNT] = $def_count;
	}

switch ( $http_vars[LINK_ACTION] )
	{
	case "Edit":
	case "NewEdit":
		PrintOneLinkToEdit ( $http_vars );
	break;
	
	case "SubmitNewLink":
		PrintOneLinkToAdd ( $http_vars );
	break;
	
	case "Submit":
		if ( SubmitLink ( $http_vars ) )
			{
			DisplayLinks ( $http_vars );
			}

		$http_vars[NEWURL] = "";
		$http_vars[TITLE] = "";
		$http_vars[DESCRIPTION] = "";
		$http_vars[LINKURL] = "";
		$http_vars[LINK_ACTION] = "";
		$http_vars[SEARCH_STRING] = "";
	break;
	
	case "Delete":
		DeleteLink ( $http_vars );
	break;
	
	case "AddNewLink":
		if ( AddNewLink ( $http_vars ) )
			{
			DisplayLinks ( $http_vars );
			}

		$http_vars[NEWURL] = "";
		$http_vars[TITLE] = "";
		$http_vars[DESCRIPTION] = "";
		$http_vars[LINKURL] = "";
		$http_vars[LINK_ACTION] = "";
		$http_vars[SEARCH_STRING] = "";
	break;
	
	default:
		if ( can_admin ( "EditLinks" ) || can_admin ( "EditCategories" ) )
			{
			// Only administrators get these choices
			switch ( $http_vars[LINK_ACTION] )
				{
				case "Activate":
					Activate ( $http_vars );
				break;
				
				case "Deactivate":
					Deactivate ( $http_vars );
				break;
				
				case "EditCategories":
					EditCategories ( $http_vars );
				break;
				
				case "SaveCategory":
					SaveCategory ( $http_vars );
				break;
				
				case "DeleteCategory":
					DeleteCategory ( $http_vars );
				break;
				
				case "DeleteFoundLinks":
					DeleteFoundLinks ( $http_vars );
				break;
				
				case "TestAllFoundLinks":
					TestAllFoundLinks ( $http_vars );
				
				default:
					DisplayLinks ( $http_vars );
				}
			}
		else
			{
			DisplayLinks ( $http_vars );
			}
	}

Links_FooterDisplay ( );
?>
